package ru.yandex.direct.chassis.entity.release.service

import org.slf4j.LoggerFactory
import org.springframework.stereotype.Component
import ru.yandex.direct.chassis.entity.release.model.ReleaseInfo
import ru.yandex.direct.chassis.util.DirectAppsConfProvider
import ru.yandex.direct.chassis.util.DirectReleaseUtils
import ru.yandex.direct.chassis.util.ReleaseVersion
import ru.yandex.direct.chassis.util.startrek.StartrekComponent
import ru.yandex.direct.chassis.util.startrek.StartrekHelper
import ru.yandex.direct.chassis.util.startrek.StartrekIssueTypeKey
import ru.yandex.direct.chassis.util.startrek.StartrekQueue
import ru.yandex.direct.chassis.util.startrek.StartrekStatusKey
import ru.yandex.startrek.client.model.Issue

/**
 * Сервис про релизы Директа
 */
@Component
class ReleaseService(
    private val startrek: StartrekHelper,
    private val directAppsConfProvider: DirectAppsConfProvider,
) {
    private val logger = LoggerFactory.getLogger(ReleaseService::class.java)

    fun getReleaseInfo(issueKey: String): ReleaseInfo? {
        val issue = startrek.findIssueByKey(issueKey)
        if (issue == null) {
            logger.warn("Release $issueKey not found")
            return null
        }
        return issueToReleaseInfo(issue)
    }

    fun getReadyToDeployReleases(): List<ReleaseInfo> {
        val issues: List<Issue> = startrek.findIssues(
            """
                Queue: ${StartrekQueue.DIRECT}
                Type: ${StartrekIssueTypeKey.RELEASE}
                Status: ${StartrekStatusKey.READY_TO_DEPLOY}
                Components: ${StartrekComponent.RELEASES_JAVA_DIRECT}
            """.trimIndent()
        )

        return issues.mapNotNull { issueToReleaseInfo(it) }
    }

    private fun issueToReleaseInfo(issue: Issue): ReleaseInfo? {
        val app = DirectReleaseUtils.getReleaseApp(issue, directAppsConfProvider.getDirectAppsConf())
        if (app == null) {
            logger.error("Unknown app for release ${issue.key}")
            return null
        }

        val version: ReleaseVersion? = DirectReleaseUtils.getReleaseVersion(issue)
        if (version == null) {
            logger.error("Invalid summary for release ${issue.key}: ${issue.summary}")
            return null
        }

        return ReleaseInfo(issue, version, app)
    }
}
