package ru.yandex.direct.chassis.entity.reports.incident;

import java.time.Instant;
import java.time.ZoneOffset;
import java.time.temporal.ChronoUnit;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import com.yandex.ydb.table.description.TableDescription;
import com.yandex.ydb.table.query.Params;
import com.yandex.ydb.table.values.ListType;
import com.yandex.ydb.table.values.PrimitiveType;
import com.yandex.ydb.table.values.PrimitiveValue;
import com.yandex.ydb.table.values.Value;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.stereotype.Component;

import ru.yandex.direct.chassis.properties.YdbSettings;
import ru.yandex.direct.chassis.util.ydb.Mapping;
import ru.yandex.direct.chassis.util.ydb.YdbClient;
import ru.yandex.direct.chassis.util.ydb.YdbTableWriter;
import ru.yandex.direct.liveresource.LiveResourceFactory;
import ru.yandex.startrek.client.Session;
import ru.yandex.startrek.client.error.ForbiddenException;
import ru.yandex.startrek.client.model.Field;
import ru.yandex.startrek.client.model.Issue;
import ru.yandex.startrek.client.model.IssueRef;
import ru.yandex.startrek.client.model.LocalLink;

import static com.yandex.ydb.table.transaction.TxControl.onlineRo;
import static com.yandex.ydb.table.transaction.TxControl.serializableRw;
import static ru.yandex.direct.chassis.configuration.StartrekConfigurationKt.MAINTENANCE_HELPERS_TRACKER_BEAN;
import static ru.yandex.direct.chassis.util.ydb.YdbClient.KEEP_IN_CACHE;

/**
 * Обновляем список Action Items тикетов, привязанных к инцидентным.
 */
@Component
public class ActionItemsReporter {
    private static final Logger logger = LoggerFactory.getLogger(ActionItemsReporter.class);

    private static final String APP = "incidents-action-items-reporter";
    private static final String UPDATED_INCIDENTS_QUERY_SOURCE = "classpath:///fetch_updated_incidents.yql";
    private static final String UPDATE_RELATIONS_QUERY_SOURCE = "classpath:///replace_incidents_action_items.yql";
    private static final String UPDATED_RELATED_ISSUES_QUERY_SOURCE =
            "classpath:///fetch_updated_incidents_related_issues.yql";

    private static final String ACTION_ITEMS_TABLE_NAME = "incidents_action_items";

    private static final List<Mapping<?, IncidentRelatedIssue, ?>> MAPPINGS = List.of(
            Mapping.readWrite(IncidentRelatedIssue::getKey, IncidentRelatedIssue::setKey)
                    .withYdbSpec("key", PrimitiveType.utf8(), PrimitiveValue::utf8)
                    .withTrackerSpec("key", Field.Schema.scalar(Field.Schema.Type.STRING, true))
                    .build(),
            Mapping.read(IncidentRelatedIssue::getIncident)
                    .withYdbSpec("incident", PrimitiveType.utf8(), PrimitiveValue::utf8)
                    .build(),
            Mapping.readWrite(IncidentRelatedIssue::getSummary, IncidentRelatedIssue::setSummary)
                    .withYdbSpec("summary", PrimitiveType.utf8(), PrimitiveValue::utf8)
                    .withTrackerSpec("summary", Field.Schema.scalar(Field.Schema.Type.STRING, true))
                    .build(),
            Mapping.readWrite(IncidentRelatedIssue::getStatus, IncidentRelatedIssue::setStatus)
                    .withYdbSpec("status", PrimitiveType.utf8(), PrimitiveValue::utf8)
                    .withTrackerSpec("status", Field.Schema.scalar(Field.Schema.Type.STATUS, true))
                    .build(),
            Mapping.readWrite(IncidentRelatedIssue::getResolution, IncidentRelatedIssue::setResolution)
                    .withYdbSpec("resolution", PrimitiveType.utf8(), PrimitiveValue::utf8)
                    .withTrackerSpec("resolution", Field.Schema.scalar(Field.Schema.Type.RESOLUTION, false))
                    .build(),
            Mapping.readWrite(IncidentRelatedIssue::getStatusChangedAtInstant, IncidentRelatedIssue::setStatusChangedAt)
                    .withYdbSpec("statusChangedAt", PrimitiveType.datetime(), PrimitiveValue::datetime)
                    .withTrackerSpec("statusStartTime", Field.Schema.scalar(Field.Schema.Type.DATETIME, false))
                    .build(),
            Mapping.readWrite(IncidentRelatedIssue::getType, IncidentRelatedIssue::setType)
                    .withYdbSpec("type", PrimitiveType.utf8(), PrimitiveValue::utf8)
                    .withTrackerSpec("type", Field.Schema.scalar(Field.Schema.Type.ISSUETYPE, true))
                    .build(),
            Mapping.readWrite(IncidentRelatedIssue::getAssignee, IncidentRelatedIssue::setAssignee)
                    .withYdbSpec("assignee", PrimitiveType.utf8(), PrimitiveValue::utf8)
                    .withTrackerSpec("assignee", Field.Schema.scalar(Field.Schema.Type.USER, false))
                    .build(),

            Mapping.readWrite(IncidentRelatedIssue::getVotes, IncidentRelatedIssue::setVotes)
                    .withYdbSpec("votes", PrimitiveType.uint64(), PrimitiveValue::uint64)
                    .withTrackerSpec("votes", Field.Schema.scalar(Field.Schema.Type.INTEGER, true))
                    .build(),
            Mapping.readWrite(IncidentRelatedIssue::getWeight, IncidentRelatedIssue::setWeight)
                    .withYdbSpec("weight", PrimitiveType.float64(), PrimitiveValue::float64)
                    .withTrackerSpec("weight", Field.Schema.scalar(Field.Schema.Type.FLOAT, false))
                    .build(),

            // нужны для вычисления признака isActionItem
            Mapping.write(IncidentRelatedIssue::addTags)
                    .withTrackerSpec("tags", Field.Schema.array(Field.Schema.Type.STRING))
                    .build(),
            Mapping.write(IncidentRelatedIssue::addComponentsRef)
                    .withTrackerSpec("components", Field.Schema.array(Field.Schema.Type.COMPONENT))
                    .build(),

            Mapping.readWrite(IncidentRelatedIssue::getCreatedAtInstant, IncidentRelatedIssue::setCreatedAt)
                    .withYdbSpec("createdAt", PrimitiveType.datetime(), PrimitiveValue::datetime)
                    .withTrackerSpec("createdAt", Field.Schema.scalar(Field.Schema.Type.DATETIME, true))
                    .build(),
            Mapping.readWrite(IncidentRelatedIssue::getUpdatedAtInstant, IncidentRelatedIssue::setUpdatedAt)
                    .withYdbSpec("updatedAt", PrimitiveType.datetime(), PrimitiveValue::datetime)
                    .withTrackerSpec("updatedAt", Field.Schema.scalar(Field.Schema.Type.DATETIME, true))
                    .build(),
            Mapping.read(IncidentRelatedIssue::getFetchedFromTrackerAt)
                    .withYdbSpec("fetchedFromTrackerAt", PrimitiveType.datetime(), PrimitiveValue::datetime)
                    .build(),

            Mapping.read(IncidentRelatedIssue::getService)
                    .withYdbSpec("service", PrimitiveType.utf8(), PrimitiveValue::utf8)
                    .build(),
            Mapping.read(IncidentRelatedIssue::getIncidentDetectTime)
                    .withYdbSpec("incidentDetectTime", PrimitiveType.datetime(), PrimitiveValue::datetime)
                    .build()
    );

    private final YdbClient ydb;
    private final YdbSettings settings;
    private final Session tracker;
    private final String queryIncidents;
    private final String queryRelated;
    private final String replaceQuery;
    private final YdbTableWriter<IncidentRelatedIssue> tableWriter;

    public ActionItemsReporter(@Qualifier(MAINTENANCE_HELPERS_TRACKER_BEAN) Session session,
                               YdbClient ydbTool,
                               YdbSettings settings
    ) {
        this.ydb = ydbTool;
        this.tracker = session;
        this.settings = settings;
        queryIncidents = LiveResourceFactory.get(UPDATED_INCIDENTS_QUERY_SOURCE).getContent();
        queryRelated = LiveResourceFactory.get(UPDATED_RELATED_ISSUES_QUERY_SOURCE).getContent();
        replaceQuery = LiveResourceFactory.get(UPDATE_RELATIONS_QUERY_SOURCE).getContent();
        tableWriter = new YdbTableWriter<>(ydb, ACTION_ITEMS_TABLE_NAME, IncidentRelatedIssue::new, MAPPINGS);
    }

    public static Map<String, Field.Schema> getCustomFields() {
        return Mapping.getCustomFields(MAPPINGS);
    }

    public void createTables() {
        TableDescription tableDescription = Mapping.getTableDescriptionBuilder(MAPPINGS)
                .setPrimaryKeys("key", "incident")
                .addGlobalIndex("byFetchedFromTrackerAt", List.of("fetchedFromTrackerAt"))
                .addGlobalIndex("byIncident", List.of("incident"))
                .build();
        ydb.createTable(ACTION_ITEMS_TABLE_NAME, tableDescription);
    }

    public void incrementalUpdateRelations() {
        logger.info("do incremental update for incidents relations");
        IncidentUpdatedAt incident = new IncidentUpdatedAt();
        incident.loadProgress();

        Params params = incident.asQueryParams();
        var reader = ydb.supplyResult(s -> s.executeDataQuery(queryIncidents, onlineRo(), params, KEEP_IN_CACHE))
                .expect("error retrieving updated incidents")
                .getResultSet(0);

        ActionItemsChecker checker = new ActionItemsChecker();
        while (reader.next()) {
            String key = reader.getColumn("key").getUtf8();
            Instant updatedAt = reader.getColumn("updatedAt").getDatetime().toInstant(ZoneOffset.UTC);
            Instant incidentDetectTime = reader.getColumn("incidentDetectTime")
                    .getDatetime()
                    .toInstant(ZoneOffset.UTC);
            String service = reader.getColumn("service").getUtf8();

            var actionItems = checker.getActionItems(key);
            replaceActionItems(service, key, incidentDetectTime, actionItems);

            incident.saveProgress(key, updatedAt);
        }
    }

    public void incrementalUpdateRelatedIssues() {
        logger.info("do incremental update for incident related issues");
        var issue = new RelatedIssueUpdatedAt();
        issue.loadProgress();

        Instant maxFetchedAt = Instant.now().minus(2, ChronoUnit.HOURS);

        Params params = Params.copyOf(issue.asQueryParams())
                .put("$maxFetchedAt", PrimitiveValue.datetime(maxFetchedAt));

        //получаем все тикеты связанные с инцидентными
        var reader = ydb.supplyResult(s -> s.executeDataQuery(queryRelated, onlineRo(), params, KEEP_IN_CACHE))
                .expect("error retrieving updated related issues")
                .getResultSet(0);

        if (reader.getRowCount() == 0) {
            // ничего не выбралось
            issue.saveProgress("", maxFetchedAt);
            return;
        }

        while (reader.next()) {
            String key = reader.getColumn("key").getUtf8();
            Instant fetchedAt = reader.getColumn("fetchedFromTrackerAt").getDatetime().toInstant(ZoneOffset.UTC);

            updateActionItem(key);

            issue.saveProgress(key, fetchedAt);
        }
    }

    private void updateActionItem(String key) {
        IncidentRelatedIssue relatedIssue;
        try {
            relatedIssue = tableWriter.parseIssue(tracker.issues().get(key));
        } catch (ForbiddenException e) {
            logger.error("Failed to load actionItem {} - no access", key);
            return;
        }

        if (!relatedIssue.isActionItem()) {
            logger.info("{} is no longer an action item — delete it from YDB", key);
            String query = "DECLARE $key AS Utf8;\n" +
                    "DELETE FROM `" + ACTION_ITEMS_TABLE_NAME + "` WHERE key = $key";
            Params params = Params.of("$key", PrimitiveValue.utf8(key));
            ydb.supplyResult(session -> session.executeDataQuery(query, serializableRw(), params, KEEP_IN_CACHE))
                    .expect("failed to delete action item " + key);
            return;
        }

        tableWriter.updateInYdb(relatedIssue, Set.of("key"), Set.of("incident", "service", "incidentDetectTime"));
    }

    private void replaceActionItems(String service, String incident, Instant incidentDetectTime,
                                    Collection<IncidentRelatedIssue> actionItems) {
        logger.info("Action items of service {} for {}: {}", service, incident, actionItems);

        actionItems.stream()
                .map(IncidentRelatedIssue::new) // делаем копию, чтобы не модифицировать элементы из кеша
                .peek(ai -> ai.setIncident(incident))
                .peek(ai -> ai.setService(service))
                .peek(ai -> ai.setIncidentDetectTime(incidentDetectTime))
                .forEach(tableWriter::saveToYdb);

        var list = actionItems.stream()
                .map(BaseIssue::getKey)
                .map(PrimitiveValue::utf8)
                .toArray(Value[]::new);

        Params params = Params.of(
                "$incident", PrimitiveValue.utf8(incident),
                "$actionItems", ListType.of(PrimitiveType.utf8()).newValueOwn(list)
        );

        ydb.supplyResult(session -> session.executeDataQuery(replaceQuery, serializableRw(), params, KEEP_IN_CACHE))
                .expect("error replacing action items for incident " + incident);
    }

    private abstract class IssueUpdatedAt {
        private final String keySettingName;
        private final String timeSettingName;
        private Instant updatedAt;
        private String key;

        private IssueUpdatedAt(String keySettingName, String timeSettingName) {
            this.keySettingName = keySettingName;
            this.timeSettingName = timeSettingName;
            updatedAt = Instant.EPOCH;
            key = "";
        }

        Params asQueryParams() {
            return Params.of(
                    "$" + timeSettingName, PrimitiveValue.datetime(updatedAt),
                    "$" + keySettingName, PrimitiveValue.utf8(key)
            );
        }

        void loadProgress() {
            logger.debug("load previously saved values for {}", this.getClass().getSimpleName());
            String lastIncident = settings.getSetting(APP, keySettingName);
            if (lastIncident != null) {
                key = lastIncident;
            }

            String lastUpdatedAt = settings.getSetting(APP, timeSettingName);
            if (lastUpdatedAt != null) {
                updatedAt = Instant.parse(lastUpdatedAt);
            }
        }

        void saveProgress(String key, Instant time) {
            this.key = key;
            this.updatedAt = time;
            settings.setSettings(APP, Map.of(keySettingName, key, timeSettingName, updatedAt.toString()));
        }
    }

    private class IncidentUpdatedAt extends IssueUpdatedAt {
        private IncidentUpdatedAt() {
            super("lastIncident", "lastUpdated");
        }
    }

    private class RelatedIssueUpdatedAt extends IssueUpdatedAt {
        private RelatedIssueUpdatedAt() {
            super("lastKey", "lastFetchedAt");
        }
    }

    private class ActionItemsChecker {
        private final Map<String, IncidentRelatedIssue> cache = new HashMap<>();

        Collection<IncidentRelatedIssue> getActionItems(String incident) {
            return failSafeLoad(incident)
                    .map(LocalLink::getObject)
                    .map(IssueRef::getKey)
                    .map(this::fetchActionItemCached)
                    .filter(ai -> ai.getKey() != null)  // те к которым нет доступа
                    .filter(IncidentRelatedIssue::isActionItem)
                    .collect(Collectors.toList());
        }

        private Stream<LocalLink> failSafeLoad(String incident) {
            try {
                return tracker.links()
                        .getLocal(incident)
                        .stream();
            } catch (ForbiddenException ignored) {
                logger.error("Failed to load relates for {} — no access", incident);
                return Stream.empty();
            }
        }

        private IncidentRelatedIssue fetchActionItemCached(String issue) {
            return cache.computeIfAbsent(issue, this::fetchActionItem);
        }

        private IncidentRelatedIssue fetchActionItem(String id) {
            try {
                Issue issue = tracker.issues().get(id);
                return tableWriter.parseIssue(issue);
            } catch (ForbiddenException ignored) {
                logger.error("Failed to load related issue {} - no access", id);
                return new IncidentRelatedIssue();
            }
        }
    }
}
