package ru.yandex.direct.chassis.entity.reports.incident;

import java.time.Instant;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.stream.Collector;
import java.util.stream.Collectors;

import javax.annotation.Nullable;

import one.util.streamex.StreamEx;

import ru.yandex.startrek.client.model.ComponentRef;
import ru.yandex.startrek.client.model.IssueTypeRef;
import ru.yandex.startrek.client.model.Ref;
import ru.yandex.startrek.client.model.ResolutionRef;
import ru.yandex.startrek.client.model.StatusRef;
import ru.yandex.startrek.client.model.UserRef;

import static ru.yandex.direct.utils.DateTimeUtils.MSK;

class BaseIssue {

    protected static final Collector<CharSequence, ?, String> JOINING = Collectors.joining(" ", " ", " ");

    private String service;
    private String queue;
    private String key;
    private String status;
    private String resolution;
    private String type;
    private String assignee;
    private String author;
    private final Set<String> tags;
    private final Set<String> components;
    private final Instant fetchedFromTrackerAt;
    private String summary;
    private String description;
    private LocalDateTime updatedAt;
    private LocalDateTime createdAt;
    private LocalDateTime statusChangedAt;
    private Long votes;
    private Double weight;

    protected BaseIssue(BaseIssue other) {
        this.tags = new HashSet<>(other.tags);
        this.components = new HashSet<>(other.components);
        this.fetchedFromTrackerAt = other.fetchedFromTrackerAt;

        this.service = other.service;
        this.queue = other.queue;
        this.key = other.key;
        this.status = other.status;
        this.resolution = other.resolution;
        this.type = other.type;
        this.assignee = other.assignee;
        this.author = other.author;
        this.summary = other.summary;
        this.description = other.description;
        this.updatedAt = other.updatedAt;
        this.createdAt = other.createdAt;
        this.statusChangedAt = other.statusChangedAt;
        this.votes = other.votes;
        this.weight = other.weight;
    }

    BaseIssue() {
        tags = new HashSet<>();
        components = new HashSet<>();
        fetchedFromTrackerAt = Instant.now();
    }

    public String getService() {
        return service;
    }

    public void setService(String service) {
        this.service = service;
    }

    void setKey(String key) {
        this.key = key;
        this.queue = key.split("-", 2)[0];
    }

    void setSummary(String summary) {
        this.summary = summary;
    }

    void setDescription(String description) {
        this.description = description;
    }

    void setStatusChangedAt(org.joda.time.Instant jodaInstant) {
        if (jodaInstant != null) {
            statusChangedAt = convert(jodaInstant);
        }
    }

    void setStatus(StatusRef statusRef) {
        this.status = statusRef.getKey();
    }

    void setResolution(ResolutionRef resolutionRef) {
        if (resolutionRef != null) {
            this.resolution = resolutionRef.getKey();
        }
    }

    void setUpdatedAt(org.joda.time.Instant jodaInstant) {
        updatedAt = convert(jodaInstant);
    }

    void setCreatedAt(org.joda.time.Instant jodaInstant) {
        createdAt = convert(jodaInstant);
    }

    void setType(IssueTypeRef issueTypeRef) {
        this.type = issueTypeRef.getKey();
    }

    void addTags(List<String> tags) {
        this.tags.addAll(tags);
    }

    void addTag(String tag) {
        tags.add(tag);
    }

    void removeTag(String tag) {
        tags.remove(tag);
    }

    void addComponentsRef(List<ComponentRef> components) {
        StreamEx.of(components)
                .map(Ref::getDisplay)
                .forEach(this.components::add);
    }

    void addComponent(String component) {
        components.add(component);
    }

    String getAssignee() {
        return assignee;
    }

    void setAssignee(UserRef userRef) {
        if (userRef != null) {
            this.assignee = userRef.getLogin();
        }
    }

    String getAuthor() {
        return author;
    }

    void setAuthor(UserRef userRef) {
        if (userRef != null) {
            this.author = userRef.getLogin();
        }
    }

    String getKey() {
        return key;
    }

    String getQueue() {
        return queue;
    }

    String getSummary() {
        return summary;
    }

    String getDescription() {
        return description;
    }

    String getType() {
        return type;
    }

    String getStatus() {
        return status;
    }

    String getResolution() {
        return resolution;
    }

    Instant getStatusChangedAtInstant() {
        return toInstant(statusChangedAt);
    }

    Long getVotes() {
        return votes;
    }

    LocalDate getUpdatedAtDate() {
        if (updatedAt == null) {
            return null;
        }
        return updatedAt.toLocalDate();
    }

    Instant getCreatedAtInstant() {
        return toInstant(createdAt);
    }

    Instant getUpdatedAtInstant() {
        return toInstant(updatedAt);
    }

    BaseIssue setVotes(Long votes) {
        this.votes = votes;
        return this;
    }

    void setWeight(Double weight) {
        this.weight = weight;
    }

    Set<String> getTags() {
        return Collections.unmodifiableSet(tags);
    }

    boolean hasTag(String tag) {
        return tags.contains(tag);
    }

    String getTagsString() {
        return concat(tags);
    }

    protected Set<String> getComponents() {
        return Collections.unmodifiableSet(components);
    }

    String getComponentsString() {
        if (components.isEmpty()) {
            return "";
        }
        return components
                .stream()
                .map(s -> s.replaceAll("\\s+", "_"))
                .sorted()
                .collect(JOINING);
    }

    Double getWeight() {
        return weight;
    }

    Instant getFetchedFromTrackerAt() {
        return fetchedFromTrackerAt;
    }

    protected LocalDateTime convert(@Nullable org.joda.time.Instant jodaInstant) {
        Instant instant = convertInstant(jodaInstant);
        if (instant == null) {
            return null;
        }
        return LocalDateTime.ofInstant(instant, MSK);
    }

    protected Instant toInstant(LocalDateTime localDateTime) {
        if (localDateTime == null) {
            return null;
        }
        return localDateTime.atZone(MSK).toInstant();
    }

    protected Instant convertInstant(@Nullable org.joda.time.Instant jodaInstant) {
        if (jodaInstant == null) {
            return null;
        }
        return Instant.ofEpochMilli(jodaInstant.getMillis());
    }

    protected String concat(Collection<String> strings) {
        if (strings.isEmpty()) {
            return "";
        }
        return strings.stream().sorted().collect(JOINING);
    }

    protected boolean hasComponent(String component) {
        return components.contains(component);
    }

    @Override
    public String toString() {
        return "BaseIssue{" +
                "key='" + key + '\'' +
                '}';
    }
}
