package ru.yandex.direct.chassis.entity.reports.incident;

import java.time.Instant;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

import one.util.streamex.StreamEx;

import ru.yandex.direct.chassis.util.startrek.StartrekComponent;
import ru.yandex.direct.chassis.util.startrek.StartrekQueue;
import ru.yandex.direct.chassis.util.startrek.StartrekTag;
import ru.yandex.startrek.client.model.ComponentRef;
import ru.yandex.startrek.client.model.Ref;
import ru.yandex.startrek.client.model.UserRef;

class Incident extends BaseIssue {
    static final String WORKING_HOURS = "!working_hours";
    static final String LONG_HOURS = "!long_hours";
    static final String WEEKEND = "!weekend";
    static final String NIGHT = "!night";
    static final String FUNCTIONAL_BUG = "#functional_bug";
    static final String CAUSED_BY_EXTERNAL_SERVICE_FAILURE = "#caused_by_external_service_failure";
    static final String IMPACT_CLIENTS_0 = "$impact_clients_0_none";
    static final String TAG_BASE_MARKUP_REQUIRED = "base_markup_required";
    static final String TAG_REQUIRED_WHAT_SEEN = "required:what_seen";
    static final String TAG_REQUIRED_CHRONOLOGY = "required:chronology";
    static final String TAG_REQUIRED_DUTY_ACTIONS = "required:duty_actions";
    static final String TAG_REQUIRED_DESCRIPTION = "required:description";
    static final String TAG_ADDED_HELP_LINK = "added_help_link";
    static final String TAG_CHECKED_BEGIN_TIME = "checked_begin_time";
    static final String TAG_ANTI_PING = "banner:noncrit";

    static final String FIGURED_OUT_FIELD_NAME = "figuredOut";
    static final String INCIDENT_REPORTER_FIELD_NAME = "incidentReporter";
    static final String SOURCE_DESCRIPTION_FIELD_NAME = "sourceDescription";
    static final String MITIGATE_FIELD_NAME = "productionDate";
    static final String INCIDENT_END_FIELD_NAME = "incidentEnd";
    static final String BEGIN_TIME_FIELD_NAME = "sreBeginTime";
    static final String HOW_TO_TEST_FIELD_NAME = "howToTest";
    static final String USER_REPORTER = "user";
    static final String SERVICE_SOURCE = "команда";
    static final String SERVICE_REPORTER = "service";
    static final String MONITORING_SOURCE = "мониторинг";
    static final String MONITORING_REPORTER = "monitoring";

    static final String DESC_WHAT_SEEN = "Что во время инцидента видели пользователи";
    static final String DESC_DESC = "Описание";
    static final String DESC_CHRONOLOGY = "Хронология событий";
    static final String DESC_DUTY_ACTIONS = "Действия дежурного";
    static final String DESC_PREVENTION_MEASURES = "Меры предотвращения";
    static final String DESC_DIAGNOSTIC = "Диагностика";
    static final String DESC_LOSSES = "Потери";

    static final Set<Long> SUPPORTED_COMPONENTS = Set.of(
            StartrekComponent.SPI_DIRECT,
            StartrekComponent.SPI_BANNER,
            StartrekComponent.SPI_METRIKA,
            StartrekComponent.SPI_PCODE,
            StartrekComponent.SPI_MODADVERT
    );

    private static final Pattern SPI_SUMMARY_PATTERN = Pattern.compile("\\[(?<date>[0-9.]+)] " +
            "(?<env>[^-]+)-(?<service>[^:]+):\n? (?<title>.+?) ?+(?:\\((?<dc>\\w+)\\))?+$");
    private static final Set<String> TAGS_STOP_LIST = Set.of(
            TAG_ADDED_HELP_LINK,
            StartrekTag.AUTO_PROCESSED_WORKING_HOURS,
            "deployed_api5",
            "deployed_binlogbroker",
            "deployed_canvas",
            "deployed_dna",
            "deployed_ess_router",
            "deployed_intapi",
            "deployed_jobs",
            "deployed_logviewer",
            "deployed_perl_direct",
            "deployed_web",
            "hide",
            "hide_from_ppalex_board",
            "incidents_notify_checked_-1001350752218",
            "incidents_notify_checked_-1001417314665",
            "incidents_notify_checked_-1001495923591",
            "incidents_notify_checked"
    );
    private static final Set<String> SPI_QUEUES = Set.of(StartrekQueue.SPI, StartrekQueue.TESTSPI);
    static final Map<String, String> IMPACT_CLIENTS = Map.of(
            IMPACT_CLIENTS_0, "нет",
            "$impact_clients_1_low", "низкое",
            "$impact_clients_2_mid", "среднее",
            "$impact_clients_3_high", "высокое");
    private static final Map<String, String> IMPACT_TEAM = Map.of(
            "$impact_team_1_low", "низкое",
            "$impact_team_2_mid", "среднее",
            "$impact_team_3_high", "высокое");
    private String source;
    private LocalDateTime beginTime;
    private org.joda.time.Instant endTime;
    private LocalDateTime detectTime;
    private org.joda.time.Instant deployTime;
    private LocalDateTime incidentEnd;
    private String figuredOut;
    private String impact;
    private String spiDC;
    private String spiEnv;
    private String spiService;
    private String spiSummary;
    private String spiTitle;
    private String incidentReporter;
    private Double vdt;
    private Double ydt;
    private List<String> duty;
    private String howToTest;
    private String supportLine;
    private List<String> sreArea;

    Incident() {
        super();
        duty = new ArrayList<>();
        sreArea = new ArrayList<>();
    }

    @Override
    void setSummary(String summary) {
        if (isSPI()) {
            Matcher matcher = SPI_SUMMARY_PATTERN.matcher(summary);
            if (matcher.find()) {
                spiSummary = summary;
                spiTitle = matcher.group("title").trim();
                summary = matcher.group("date").replace('.', '-') + ": " + matcher.group("title");
            }
        }
        super.setSummary(summary);
    }

    static Map<String, String> parseSpiDescription(String description) {
        Map<String, String> result = new HashMap<>();
        String currentBlock = null;
        StringBuilder content = null;
        for (String s : description.split("\n")) {
            if (s.startsWith("===") || s.startsWith("##")) {
                String name = s.replaceAll("(?:^[=#]+|[=#]+$)", "").trim();
                if (currentBlock != null) {
                    result.put(currentBlock, content.toString());
                }
                currentBlock = name;
                content = new StringBuilder();
                continue;
            }
            if (currentBlock == null) {
                continue;
            }
            if (s.startsWith("++//!!") || s.startsWith("*++{")) {
                continue;
            }
            String line = s.replace("&nbsp;", "").trim();
            if (!line.isEmpty()) {
                if (content.length() > 0) {
                    content.append("\n");
                }
                content.append(line);
            }
        }
        if (currentBlock != null) {
            result.put(currentBlock, content.toString());
        }
        return result;
    }

    void setBeginTime(org.joda.time.Instant jodaInstant) {
        beginTime = convert(jodaInstant);
    }

    void setDetectTime(org.joda.time.Instant jodaInstant) {
        detectTime = convert(jodaInstant);
    }

    void setDeployTime(org.joda.time.Instant jodaInstant) {
        deployTime = jodaInstant;
    }

    void setEndTime(org.joda.time.Instant jodaInstant) {
        endTime = jodaInstant;
    }

    void setIncidentEnd(org.joda.time.Instant jodaInstant) {
        incidentEnd = convert(jodaInstant);
    }

    void setTags(List<String> tags) {
        for (String tag : tags) {
            // Костыль переезда в SPI: все что выглядит как наша разметка — считаем компонентами
            if (isSPI() && tag.startsWith(StartrekTag.SPI_PREFIX)) {
                String component = tag.substring(StartrekTag.SPI_PREFIX.length());
                addComponent(component);
                continue;
            }
            addTag(tag);
        }

        Map<String, List<String>> tagKV = StreamEx.of(getTags())
                .map(t -> t.split(":"))
                .filter(a -> a.length == 2)
                .mapToEntry(a -> a[0], a -> a[1])
                .grouping();

        List<String> dc = tagKV.get("dc");
        if (dc != null && dc.size() == 1) {
            spiDC = dc.get(0).toUpperCase();
        }
        List<String> env = tagKV.get("env");
        if (env != null && env.size() == 1) {
            spiEnv = env.get(0).toUpperCase();
        }
        List<String> service = tagKV.get("service");
        if (service != null && service.size() == 1) {
            spiService = service.get(0);
        }
    }

    @Override
    String getTagsString() {
        return concat(getTags());
    }

    String getSpiService() {
        return spiService;
    }

    @Override
    Set<String> getTags() {
        return StreamEx.of(super.getTags())
                .remove(TAGS_STOP_LIST::contains)
                .toSet();
    }

    @Override
    void addComponentsRef(List<ComponentRef> components) {
        if (isSPI()) {
            // в SPI в компонентах — только пострадавший сервис.
            // мы такое обычно размечали как @direct — имитируем её вручную
            components.stream()
                    .filter(c -> SUPPORTED_COMPONENTS.contains(c.getId()))
                    .map(Ref::getDisplay)
                    .findAny()
                    .ifPresent(s -> {
                        setService(s);
                        addComponent("@" + s);
                    });
        } else {
            super.addComponentsRef(components);
            super.setService("direct");
        }
    }

    private String getServiceToSave(String service) {
        if (service == null) {
            return "other";
        }
        return StartrekQueue.DIRECTINCIDENTS.equals(getQueue()) ? "direct" : service;
    }

    String getSource() {
        return source;
    }

    void setSource(String source) {
        this.source = source;
    }

    String getFiguredOut() {
        return figuredOut;
    }

    void setFiguredOut(String figuredOut) {
        this.figuredOut = figuredOut;
    }

    String getImpact() {
        return impact;
    }

    void setImpact(String impact) {
        this.impact = impact;
    }

    void setIncidentReporter(String reporter) {
        this.incidentReporter = reporter;
    }

    void setVdt(Double vdt) {
        this.vdt = vdt;
    }

    Double getVdt() {
        return vdt;
    }

    void setYdt(Double ydt) {
        this.ydt = ydt;
    }

    Double getYdt() {
        return ydt;
    }

    List<String> getDuty() {
        return duty;
    }

    void setDuty(List<UserRef> duties) {
        duties.stream().map(UserRef::getLogin).forEach(duty::add);
    }

    List<String> getSreArea() {
        return sreArea;
    }

    void setSreArea(List<String> sreArea) {
        this.sreArea.addAll(sreArea);
    }

    String getHowToTest() {
        return howToTest;
    }

    void setHowToTest(String howToTest) {
        this.howToTest = howToTest;
    }

    String getSupportLine() {
        return supportLine;
    }

    void setSupportLine(String supportLine) {
        this.supportLine = supportLine;
    }

    String getIncidentReporter() {
        return incidentReporter;
    }

    String getImpactClients() {
        return getMappedComponent(IMPACT_CLIENTS);
    }

    String getImpactTeam() {
        return getMappedComponent(IMPACT_TEAM);
    }

    boolean isFunctionalBug() {
        return hasComponent(FUNCTIONAL_BUG);
    }

    boolean isCausedByExternalServiceFailure() {
        return hasComponent(CAUSED_BY_EXTERNAL_SERVICE_FAILURE)
                || hasTag(StartrekTag.SPI_VICTIM);
    }

    Instant getBeginTimeInstant() {
        return toInstant(beginTime);
    }

    Instant getDetectTimeInstant() {
        return toInstant(detectTime);
    }

    Instant getDeployTimeInstant() {
        return convertInstant(deployTime);
    }

    org.joda.time.Instant getDeployTime() {
        return deployTime;
    }

    Instant getEndTimeInstant() {
        return convertInstant(endTime);
    }

    org.joda.time.Instant getEndTime() {
        return endTime;
    }

    Instant getIncidentEnd() {
        return toInstant(incidentEnd);
    }

    boolean isSPI() {
        return SPI_QUEUES.contains(getQueue());
    }

    String composeSpiSummary() {
        if (spiTitle == null
                || detectTime == null
                || spiService == null
                || spiEnv == null
        ) {
            return null;
        }

        var sb = new StringBuilder();
        sb.append('[');
        sb.append(detectTime.format(DateTimeFormatter.ofPattern("yyyy.MM.dd")));
        sb.append("] ");
        sb.append(spiEnv);
        sb.append('-');
        sb.append("DIRECT");
        if (!spiService.equals("ALL")) {
            sb.append(" / ");
            sb.append(spiService);
        }
        sb.append(": ");
        sb.append(spiTitle);
        if (spiDC != null) {
            sb.append(" (");
            sb.append(spiDC);
            sb.append(')');
        }
        return sb.toString();
    }

    String getSpiSummary() {
        return spiSummary;
    }

    private String getMappedComponent(Map<String, String> mapping) {
        List<String> candidates = getComponents()
                .stream()
                .filter(mapping::containsKey)
                .map(mapping::get)
                .collect(Collectors.toList());

        if (candidates.size() != 1) {
            return null;
        }
        return candidates.get(0);
    }

    @Override
    public String toString() {
        return "Incident{" +
                "key='" + getKey() + '\'' +
                '}';
    }
}
