package ru.yandex.direct.chassis.entity.reports.incident;

import java.time.DayOfWeek;
import java.time.Duration;
import java.time.Instant;
import java.time.LocalDate;
import java.time.LocalTime;
import java.time.format.DateTimeFormatter;
import java.time.format.DateTimeParseException;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.function.BiConsumer;
import java.util.stream.Collectors;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import com.google.common.base.Joiner;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import com.yandex.ydb.table.description.TableDescription;
import com.yandex.ydb.table.values.PrimitiveType;
import com.yandex.ydb.table.values.PrimitiveValue;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.direct.chassis.properties.YdbSettings;
import ru.yandex.direct.chassis.util.startrek.StartrekComponent;
import ru.yandex.direct.chassis.util.startrek.StartrekQueue;
import ru.yandex.direct.chassis.util.startrek.StartrekTag;
import ru.yandex.direct.chassis.util.ydb.Mapping;
import ru.yandex.direct.chassis.util.ydb.YdbClient;
import ru.yandex.direct.chassis.util.ydb.YdbTableWriter;
import ru.yandex.direct.scheduler.Hourglass;
import ru.yandex.direct.scheduler.support.DirectJob;
import ru.yandex.direct.utils.DateTimeUtils;
import ru.yandex.startrek.client.Session;
import ru.yandex.startrek.client.error.EntityNotFoundException;
import ru.yandex.startrek.client.error.ForbiddenException;
import ru.yandex.startrek.client.model.CommentCreate;
import ru.yandex.startrek.client.model.CommentUpdate;
import ru.yandex.startrek.client.model.Component;
import ru.yandex.startrek.client.model.Field;
import ru.yandex.startrek.client.model.IssueRef;
import ru.yandex.startrek.client.model.IssueUpdate;
import ru.yandex.startrek.client.model.LocalLink;

import static java.time.format.DateTimeFormatter.ISO_LOCAL_DATE;
import static org.apache.commons.lang3.StringUtils.isBlank;
import static org.apache.commons.lang3.StringUtils.isNotBlank;
import static ru.yandex.direct.chassis.configuration.StartrekConfigurationKt.MAINTENANCE_HELPERS_TRACKER_BEAN;
import static ru.yandex.direct.chassis.entity.reports.incident.Incident.BEGIN_TIME_FIELD_NAME;
import static ru.yandex.direct.chassis.entity.reports.incident.Incident.DESC_CHRONOLOGY;
import static ru.yandex.direct.chassis.entity.reports.incident.Incident.DESC_DESC;
import static ru.yandex.direct.chassis.entity.reports.incident.Incident.DESC_DIAGNOSTIC;
import static ru.yandex.direct.chassis.entity.reports.incident.Incident.DESC_DUTY_ACTIONS;
import static ru.yandex.direct.chassis.entity.reports.incident.Incident.DESC_LOSSES;
import static ru.yandex.direct.chassis.entity.reports.incident.Incident.DESC_PREVENTION_MEASURES;
import static ru.yandex.direct.chassis.entity.reports.incident.Incident.DESC_WHAT_SEEN;
import static ru.yandex.direct.chassis.entity.reports.incident.Incident.FIGURED_OUT_FIELD_NAME;
import static ru.yandex.direct.chassis.entity.reports.incident.Incident.HOW_TO_TEST_FIELD_NAME;
import static ru.yandex.direct.chassis.entity.reports.incident.Incident.IMPACT_CLIENTS_0;
import static ru.yandex.direct.chassis.entity.reports.incident.Incident.INCIDENT_END_FIELD_NAME;
import static ru.yandex.direct.chassis.entity.reports.incident.Incident.INCIDENT_REPORTER_FIELD_NAME;
import static ru.yandex.direct.chassis.entity.reports.incident.Incident.LONG_HOURS;
import static ru.yandex.direct.chassis.entity.reports.incident.Incident.MITIGATE_FIELD_NAME;
import static ru.yandex.direct.chassis.entity.reports.incident.Incident.NIGHT;
import static ru.yandex.direct.chassis.entity.reports.incident.Incident.SOURCE_DESCRIPTION_FIELD_NAME;
import static ru.yandex.direct.chassis.entity.reports.incident.Incident.SUPPORTED_COMPONENTS;
import static ru.yandex.direct.chassis.entity.reports.incident.Incident.TAG_ADDED_HELP_LINK;
import static ru.yandex.direct.chassis.entity.reports.incident.Incident.TAG_ANTI_PING;
import static ru.yandex.direct.chassis.entity.reports.incident.Incident.TAG_BASE_MARKUP_REQUIRED;
import static ru.yandex.direct.chassis.entity.reports.incident.Incident.TAG_CHECKED_BEGIN_TIME;
import static ru.yandex.direct.chassis.entity.reports.incident.Incident.TAG_REQUIRED_CHRONOLOGY;
import static ru.yandex.direct.chassis.entity.reports.incident.Incident.TAG_REQUIRED_DESCRIPTION;
import static ru.yandex.direct.chassis.entity.reports.incident.Incident.TAG_REQUIRED_WHAT_SEEN;
import static ru.yandex.direct.chassis.entity.reports.incident.Incident.WEEKEND;
import static ru.yandex.direct.chassis.entity.reports.incident.Incident.WORKING_HOURS;
import static ru.yandex.direct.chassis.entity.reports.incident.Incident.parseSpiDescription;
import static ru.yandex.direct.chassis.util.StartrekTools.signedMessageFactory;
import static ru.yandex.direct.utils.CommonUtils.ifNotNull;
import static ru.yandex.direct.utils.CommonUtils.nvl;

/**
 * Импорт инцидентов из трекера, обработка и загрузка их в YDB для дашборда и графиков.
 */
@Hourglass(periodInSeconds = 600)
public class IncidentsReporter extends DirectJob {
    static final String BY_DETECT_TIME_IDX = "byDetectTime";
    static final String BY_SERVICE_IDX = "byService";
    static final String SLACK_LINK = "((https://yndx-ad-tech.slack.com/archives/C01GZFAR3RT Slack: direct-incident))";
    private static final Logger logger = LoggerFactory.getLogger(IncidentsReporter.class);
    private static final String DIRECT_SERVICE_NAME = "direct";
    private static final String APP = "incidents-stat";
    private static final String YDB_TABLE = "incidents";
    private static final String SETTING_MAX_UPDATED_AT = "maxUpdatedAt";
    private static final String SELF = "https://a.yandex-team.ru/arc/trunk/arcadia/direct/apps/maintenance-helpers" +
            "/src/main/java/ru/yandex/direct/reports/incident/IncidentsReporter.java";

    private static final int START_FROM = 489;
    private static final String QUERY_OLD = "Queue: " + StartrekQueue.DIRECTINCIDENTS
            + " AND Key: >= " + StartrekQueue.DIRECTINCIDENTS + "-" + START_FROM
            + " AND Components: !" + StartrekComponent.INCIDENTS_NONE;
    private static final String QUERY_NEW = "Queue: " + StartrekQueue.SPI
            + " Components: " + Joiner.on(",").join(SUPPORTED_COMPONENTS)
            + " Tags: !\"direct:@none\"";
    private static final String QUERY = "( (" + QUERY_OLD + ") OR (" + QUERY_NEW + ") )";

    private static final List<Mapping<?, Incident, ?>> MAPPINGS = List.of(
            Mapping.readWrite(Incident::getKey, Incident::setKey)
                    .withYdbSpec("key", PrimitiveType.utf8(), PrimitiveValue::utf8)
                    .withTrackerSpec("key", Field.Schema.scalar(Field.Schema.Type.STRING, true))
                    .build(),
            Mapping.readWrite(Incident::getSummary, Incident::setSummary)
                    .withYdbSpec("summary", PrimitiveType.utf8(), PrimitiveValue::utf8)
                    .withTrackerSpec("summary", Field.Schema.scalar(Field.Schema.Type.STRING, true))
                    .build(),
            Mapping.write(Incident::setDescription)
                    .withTrackerSpec("description", Field.Schema.scalar(Field.Schema.Type.TEXT, false))
                    .build(),
            Mapping.readWrite(Incident::getStatus, Incident::setStatus)
                    .withYdbSpec("status", PrimitiveType.utf8(), PrimitiveValue::utf8)
                    .withTrackerSpec("status", Field.Schema.scalar(Field.Schema.Type.STATUS, true))
                    .build(),
            Mapping.write(Incident::setResolution)
                    .withTrackerSpec("resolution", Field.Schema.scalar(Field.Schema.Type.RESOLUTION, false))
                    .build(),
            Mapping.readWrite(Incident::getSource, Incident::setSource)
                    .withYdbSpec("source", PrimitiveType.utf8(), PrimitiveValue::utf8)
                    .withTrackerSpec(SOURCE_DESCRIPTION_FIELD_NAME,
                            Field.Schema.scalar(Field.Schema.Type.STRING, false))
                    .build(),
            Mapping.readWrite(Incident::getVdt, Incident::setVdt)
                    .withYdbSpec("vdt", PrimitiveType.float64(), PrimitiveValue::float64)
                    .withTrackerSpec("vdt", Field.Schema.scalar(Field.Schema.Type.FLOAT, false))
                    .build(),
            Mapping.readWrite(Incident::getYdt, Incident::setYdt)
                    .withYdbSpec("ydt", PrimitiveType.float64(), PrimitiveValue::float64)
                    .withTrackerSpec("ydt", Field.Schema.scalar(Field.Schema.Type.FLOAT, false))
                    .build(),

            Mapping.readWrite(Incident::getUpdatedAtInstant, Incident::setUpdatedAt)
                    .withYdbSpec("updatedAt", PrimitiveType.datetime(), PrimitiveValue::datetime)
                    .withTrackerSpec("updatedAt", Field.Schema.scalar(Field.Schema.Type.DATETIME, true))
                    .build(),
            Mapping.readWrite(Incident::getCreatedAtInstant, Incident::setCreatedAt)
                    .withYdbSpec("createdAt", PrimitiveType.datetime(), PrimitiveValue::datetime)
                    .withTrackerSpec("createdAt", Field.Schema.scalar(Field.Schema.Type.DATETIME, true))
                    .build(),
            Mapping.readWrite(Incident::getBeginTimeInstant, Incident::setBeginTime)
                    .withYdbSpec("beginTime", PrimitiveType.datetime(), PrimitiveValue::datetime)
                    .withTrackerSpec(BEGIN_TIME_FIELD_NAME, Field.Schema.scalar(Field.Schema.Type.DATETIME, false))
                    .build(),
            Mapping.readWrite(Incident::getDetectTimeInstant, Incident::setDetectTime)
                    .withYdbSpec("detectTime", PrimitiveType.datetime(), PrimitiveValue::datetime)
                    .withTrackerSpec("notificationTime", Field.Schema.scalar(Field.Schema.Type.DATETIME, false))
                    .build(),
            Mapping.readWrite(Incident::getDeployTimeInstant, Incident::setDeployTime)
                    .withYdbSpec("deployTime", PrimitiveType.datetime(), PrimitiveValue::datetime)
                    .withTrackerSpec(MITIGATE_FIELD_NAME, Field.Schema.scalar(Field.Schema.Type.DATETIME, false))
                    .build(),
            Mapping.readWrite(Incident::getEndTimeInstant, Incident::setEndTime)
                    .withYdbSpec("endTime", PrimitiveType.datetime(), PrimitiveValue::datetime)
                    .withTrackerSpec("sreEndTime", Field.Schema.scalar(Field.Schema.Type.DATETIME, false))
                    .build(),
            Mapping.readWrite(Incident::getIncidentEnd, Incident::setIncidentEnd)
                    .withYdbSpec(INCIDENT_END_FIELD_NAME, PrimitiveType.datetime(), PrimitiveValue::datetime)
                    .withTrackerSpec(INCIDENT_END_FIELD_NAME, Field.Schema.scalar(Field.Schema.Type.DATETIME, false))
                    .build(),

            Mapping.write(Incident::setIncidentReporter)
                    .withTrackerSpec(INCIDENT_REPORTER_FIELD_NAME, Field.Schema.scalar(Field.Schema.Type.STRING, false))
                    .build(),
            Mapping.readWrite(Incident::getImpact, Incident::setImpact)
                    .withYdbSpec("impact", PrimitiveType.utf8(), PrimitiveValue::utf8)
                    .withTrackerSpec("impact", Field.Schema.scalar(Field.Schema.Type.STRING, false))
                    .build(),
            Mapping.read(Incident::getImpactClients)
                    .withYdbSpec("impactClients", PrimitiveType.utf8(), PrimitiveValue::utf8)
                    .build(),
            Mapping.read(Incident::getImpactTeam)
                    .withYdbSpec("impactTeam", PrimitiveType.utf8(), PrimitiveValue::utf8)
                    .build(),
            Mapping.read(Incident::isFunctionalBug)
                    .withYdbSpec("functionalBug", PrimitiveType.bool(), PrimitiveValue::bool)
                    .build(),
            Mapping.read(Incident::isCausedByExternalServiceFailure)
                    .withYdbSpec("causedByExternalServiceFailure", PrimitiveType.bool(), PrimitiveValue::bool)
                    .build(),

            Mapping.readWrite(Incident::getTagsString, Incident::setTags)
                    .withYdbSpec("tags", PrimitiveType.utf8(), PrimitiveValue::utf8)
                    .withTrackerSpec("tags", Field.Schema.array(Field.Schema.Type.STRING))
                    .build(),
            Mapping.readWrite(Incident::getComponentsString, Incident::addComponentsRef)
                    .withYdbSpec("components", PrimitiveType.utf8(), PrimitiveValue::utf8)
                    .withTrackerSpec("components", Field.Schema.array(Field.Schema.Type.COMPONENT))
                    .build(),
            Mapping.read(Incident::getService)
                    .withYdbSpec("service", PrimitiveType.utf8(), PrimitiveValue::utf8)
                    .build(),
            Mapping.read(Incident::getSpiService)
                    .withYdbSpec("component", PrimitiveType.utf8(), PrimitiveValue::utf8)
                    .build(),
            Mapping.readWrite(Incident::getFiguredOut, Incident::setFiguredOut)
                    .withYdbSpec(FIGURED_OUT_FIELD_NAME, PrimitiveType.utf8(), PrimitiveValue::utf8)
                    .withTrackerSpec(FIGURED_OUT_FIELD_NAME, Field.Schema.scalar(Field.Schema.Type.STRING, false))
                    .build(),

            Mapping.write(Incident::setAuthor)
                    .withTrackerSpec("author", Field.Schema.scalar(Field.Schema.Type.USER, false))
                    .build(),
            Mapping.write(Incident::setAssignee)
                    .withTrackerSpec("assignee", Field.Schema.scalar(Field.Schema.Type.USER, false))
                    .build(),
            Mapping.write(Incident::setDuty)
                    .withTrackerSpec("duty", Field.Schema.array(Field.Schema.Type.USER))
                    .build(),
            Mapping.write(Incident::setSreArea)
                    .withTrackerSpec("SREArea", Field.Schema.array(Field.Schema.Type.STRING))
                    .build(),
            Mapping.write(Incident::setHowToTest)
                    .withTrackerSpec(HOW_TO_TEST_FIELD_NAME, Field.Schema.scalar(Field.Schema.Type.STRING, false))
                    .build(),
            Mapping.write(Incident::setSupportLine)
                    .withTrackerSpec("supportLine", Field.Schema.scalar(Field.Schema.Type.STRING, false))
                    .build(),
            Mapping.read(Incident::getFetchedFromTrackerAt)
                    .withYdbSpec("fetchedFromTrackerAt", PrimitiveType.datetime(), PrimitiveValue::datetime)
                    .build()
    );

    private final ru.yandex.startrek.client.Session client;
    private final YdbClient ydb;
    private final YdbSettings settings;
    private final YdbTableWriter<Incident> tableWriter;
    private final LoadingCache<String, Component> componentsLoader;
    private final boolean isWritingStartrekActionsAllowed;


    public IncidentsReporter(
            @Qualifier(MAINTENANCE_HELPERS_TRACKER_BEAN) Session client,
            YdbClient ydb,
            YdbSettings settings,
            @Value("${maintenance-helpers.writing_startrek_actions_allowed}") boolean isWritingStartrekActionsAllowed
    ) {
        this.client = client;
        this.ydb = ydb;
        this.settings = settings;
        this.tableWriter = new YdbTableWriter<>(ydb, YDB_TABLE, Incident::new, MAPPINGS);
        this.isWritingStartrekActionsAllowed = isWritingStartrekActionsAllowed;
        componentsLoader = CacheBuilder.newBuilder()
                .maximumSize(100)
                .expireAfterWrite(Duration.ofHours(1))
                .build(CacheLoader.from(this::getIncidentsComponent));
    }

    public static Map<String, Field.Schema> getCustomFields() {
        return Mapping.getCustomFields(MAPPINGS);
    }

    private static String formatTime(Instant instant) {
        if (instant == null) {
            return "не заполнено";
        }
        var localDateTime = DateTimeUtils.instantToMoscowDateTime(instant);
        return DateTimeFormatter.ofPattern("MM/dd HH:mm").format(localDateTime);
    }

    public void createTables() {
        TableDescription tableDescription = Mapping.getTableDescriptionBuilder(MAPPINGS)
                .setPrimaryKey("key")
                .addGlobalIndex("byUpdatedAt", List.of("updatedAt"))
                .addGlobalIndex(BY_DETECT_TIME_IDX, List.of("detectTime"))
                .addGlobalIndex(BY_SERVICE_IDX, List.of("service"))
                .build();
        ydb.createTable(YDB_TABLE, tableDescription);
    }

    private LocalDate loadProgress() {
        logger.debug("load previously saved updatedAt value");

        String lastValue = settings.getSetting(APP, SETTING_MAX_UPDATED_AT);
        if (lastValue == null) {
            logger.debug("there is no saved updatedAt");
            return LocalDate.EPOCH;
        }

        try {
            return LocalDate.parse(lastValue, ISO_LOCAL_DATE);
        } catch (DateTimeParseException e) {
            logger.error("Failed to parse saved value", e);
            return LocalDate.EPOCH;
        }
    }

    @Override
    public void execute() {
        LocalDate lastMaxUpdatedAt = loadProgress();
        String query = QUERY + " AND Updated: >= " + ISO_LOCAL_DATE.format(lastMaxUpdatedAt);
        LocalDate newMaxUpdatedAt = processIncidentsByQuery(query);
        saveProgress(newMaxUpdatedAt);
    }

    @Nullable
    private LocalDate processIncidentsByQuery(String query) {
        LocalDate maxUpdatedAt = null;

        logger.info("fetch incidents by query {}", query);
        var iterator = client.issues()
                .find(query)
                .map(tableWriter::parseIssue);

        while (iterator.hasNext()) {
            Incident incident = iterator.next();
            LocalDate updatedAt = incident.getUpdatedAtDate();
            if (maxUpdatedAt == null || updatedAt != null && updatedAt.compareTo(maxUpdatedAt) > 0) {
                maxUpdatedAt = updatedAt;
            }

            var update = incidentAutoActions(incident);
            if (!update.isEmpty()) {
                String incidentKey = incident.getKey();
                var updateValues = update.getUpdate().getValues();
                if (!isWritingStartrekActionsAllowed) {
                    logger.info("dry run, updating issue {}: {}", incidentKey, updateValues);
                } else {
                    try {
                        logger.info("Apply auto action update to {}: {}", incidentKey, updateValues);
                        client.issues().update(incidentKey, update.getUpdate(), update.isNotify(), update.isNotify());
                    } catch (RuntimeException e) {
                        logger.error("Failed to apply incident update", e);
                    }
                }
            }

            tableWriter.saveToYdb(incident);
        }

        return maxUpdatedAt;
    }

    private IncidentUpdate incidentAutoActions(Incident incident) {
        if (!DIRECT_SERVICE_NAME.equalsIgnoreCase(incident.getService())) {
            UpdateContainer container = new UpdateContainer();
            return container.buildIncidentUpdate();
        }

        List<BiConsumer<Incident, UpdateContainer>> actions = List.of(
                this::removeBeginTimeIfItEqualsToNotify,
                this::inviteDutyToMarkUp,
                this::addAntiPingTag,
                this::processBaseMarkupRequiredTag,
                this::processSolvedSpi,
                this::fixSpiTitleFromTags,
                this::fillIncidentReporterFromSource,
                this::fillSourceFromIncidentReporter,
                this::setZeroClientsImpactForInternalIncidents,
                this::autoWorkingHours,
                this::progressBarComment
        );
        int i = 0;
        UpdateContainer container = new UpdateContainer();
        for (var action : actions) {
            try {
                i++;
                action.accept(incident, container);
            } catch (RuntimeException e) {
                logger.error("Failed to proceed auto action #" + i + " for " + incident.getKey(), e);
            }
        }
        return container.buildIncidentUpdate();
    }

    private void addAntiPingTag(Incident incident, UpdateContainer container) {
        if (!incident.hasTag("service:NP")
                || incident.hasTag(TAG_ANTI_PING)) {
            return;
        }
        container.addTag(TAG_ANTI_PING);
        incident.addTag(TAG_ANTI_PING);
    }

    private void removeBeginTimeIfItEqualsToNotify(Incident incident, UpdateContainer container) {
        if (incident.hasTag("flow:incident_started") || incident.hasTag(TAG_CHECKED_BEGIN_TIME)) {
            return;
        }
        Instant beginTimeInstant = incident.getBeginTimeInstant();
        Instant detectTimeInstant = incident.getDetectTimeInstant();
        Instant endTimeInstant = incident.getEndTimeInstant();

        container.addTag(TAG_CHECKED_BEGIN_TIME);

        if (beginTimeInstant == null
                || detectTimeInstant == null
                || beginTimeInstant.equals(endTimeInstant)
        ) {
            return;
        }
        if (Objects.equals(beginTimeInstant, detectTimeInstant)) {
            container.set(BEGIN_TIME_FIELD_NAME, null);
            incident.setBeginTime(null);
        }
    }

    private void autoWorkingHours(Incident incident, UpdateContainer container) {
        var incidentEnd = ifNotNull(incident.getEndTimeInstant(), DateTimeUtils::instantToMoscowDateTime);
        var detect = ifNotNull(incident.getDetectTimeInstant(), DateTimeUtils::instantToMoscowDateTime);

        if (incident.hasTag(StartrekTag.AUTO_PROCESSED_WORKING_HOURS)
                || detect == null
                || incidentEnd == null
        ) {
            return;
        }

        final LocalTime t07h = LocalTime.of(7, 0);
        final LocalTime t10h = LocalTime.of(10, 0);
        final LocalTime t21h = LocalTime.of(21, 0);

        DayOfWeek detectDOW = detect.getDayOfWeek();
        DayOfWeek endDOW = incidentEnd.getDayOfWeek();
        LocalTime detectTime = detect.toLocalTime();
        LocalTime endTime = incidentEnd.toLocalTime();
        // можно еще смотреть на времена mitigate и окончания влияния
        // но их трактовка спорна, поэтому пока не делаем

        Set<String> toRemove = new HashSet<>(List.of(WORKING_HOURS, LONG_HOURS, NIGHT, WEEKEND));
        Set<String> toAdd;

        boolean isWeekend = endDOW == DayOfWeek.SATURDAY || endDOW == DayOfWeek.SUNDAY
                || detectDOW == DayOfWeek.SATURDAY || detectDOW == DayOfWeek.SUNDAY;
        boolean isNight = endTime.isBefore(t07h) || detectTime.isBefore(t07h);
        boolean isMorningOrEvening = endTime.isAfter(t21h) || endTime.isBefore(t10h)
                || detectTime.isAfter(t21h) || detectTime.isBefore(t10h);

        if (isNight && isWeekend) {
            toAdd = Set.of(WEEKEND, NIGHT);
        } else if (isWeekend) {
            toAdd = Set.of(WEEKEND);
        } else if (isNight) {
            toAdd = Set.of(NIGHT);
        } else {
            if (isMorningOrEvening) {
                toAdd = Set.of(LONG_HOURS);
            } else {
                toAdd = Set.of(WORKING_HOURS);
            }
        }
        toRemove.removeAll(toAdd);

        boolean allAdded = toAdd.stream().allMatch(incident::hasComponent);
        boolean allRemoved = toRemove.stream().noneMatch(incident::hasComponent);
        if (allAdded && allRemoved) {
            return;
        }

        logger.info("Mark {} as {}", incident.getKey(), toAdd);
        container.addComment("Размечено как: " + String.join(", ", toAdd) + ". Неверно? Действуй " +
                "((https://docs.yandex-team.ru/direct-dev/incidents/spi-ticket#hours по инструкции))");

        if (incident.isSPI()) {
            toAdd.stream().map(IncidentsReporter::spiTag).forEach(container::addTag);
            toRemove.stream().map(IncidentsReporter::spiTag).forEach(container::removeTag);
        } else {
            toAdd.stream().map(componentsLoader::getUnchecked).forEach(container::addComponent);
            toRemove.stream().map(componentsLoader::getUnchecked).forEach(container::removeComponent);
        }
    }

    void progressBarComment(Incident incident, UpdateContainer container) {
        String commentId = incident.getHowToTest();
        if (!incident.isSPI()) {
            return;
        }

        String newText = makeWhatToFillCommentAndTags(incident, container);
        if ("-".equals(commentId)) {
            return; // комментарий отключен или сломался, только теги
        }

        String incidentKey = incident.getKey();

        if (commentId == null) {
            if (!isWritingStartrekActionsAllowed) {
                logger.info("dry run, commenting issue {}: {}", incidentKey, newText);
                return;
            }
            long id = client.comments()
                    .create(incidentKey, CommentCreate.comment(newText).build(), false, false)
                    .getId();
            String strId = String.valueOf(id);
            container.set(HOW_TO_TEST_FIELD_NAME, strId);
            incident.setHowToTest(strId);
            return;
        }

        long id;
        try {
            id = Long.parseLong(commentId);
        } catch (NumberFormatException e) {
            logger.debug("can't parse status-comment id in {}", incidentKey);
            container.set(HOW_TO_TEST_FIELD_NAME, "-");
            incident.setHowToTest("-");
            return;
        }

        if (!isWritingStartrekActionsAllowed) {
            logger.info("dry run, updating comment {}: {}", incidentKey, newText);
            return;
        }
        String oldText;
        try {
            oldText = client.comments().get(incidentKey, id).getText().get();
        } catch (EntityNotFoundException e) {
            logger.warn("status comment {} not found in {}. disable tracking", id, incidentKey);
            container.set(HOW_TO_TEST_FIELD_NAME, "-");
            incident.setHowToTest("-");
            return;
        }

        if (newText.equals(oldText)) {
            return;
        }
        client.comments().update(incidentKey, id, CommentUpdate.comment(newText).build(), false, false);
    }

    private Component getIncidentsComponent(String name) {
        return client.components()
                .getAll(StartrekQueue.DIRECTINCIDENTS)
                .stream()
                .filter(component -> component.getName().equals(name))
                .findFirst()
                .orElseThrow();
    }

    private void processBaseMarkupRequiredTag(Incident incident, UpdateContainer container) {
        boolean spiPart = false;
        if (incident.isSPI()) {
            Map<String, String> parsed = parseSpiDescription(incident.getDescription());
            String[] fields = {DESC_WHAT_SEEN, DESC_DESC, DESC_CHRONOLOGY};
            for (String field : fields) {
                if (parsed.getOrDefault(field, "").length() < 10) {
                    spiPart = true;
                }
            }
        }
        boolean currentHasTag = incident.hasTag(TAG_BASE_MARKUP_REQUIRED);
        boolean desiredHasTag = incident.getImpactTeam() == null
                || incident.getImpact() == null
                || incident.getDetectTimeInstant() == null
                || isBlank(incident.getSource())
                || spiPart
                || incident.getEndTimeInstant() == null;
        if (currentHasTag == desiredHasTag) {
            return;
        }

        logger.info("Update incident {} {} tag state: current {}, desired {}",
                incident.getKey(), TAG_BASE_MARKUP_REQUIRED, currentHasTag, desiredHasTag);

        if (desiredHasTag) {
            container.addTag(TAG_BASE_MARKUP_REQUIRED);
            incident.addTag(TAG_BASE_MARKUP_REQUIRED);
            if (incident.isSPI()) {
                // в SPI переходим на полное оформление по методичке
                return;
            }
            container.addComment("После завершения аварии, пожалуйста, заполни " +
                    " ((https://docs.yandex-team.ru/direct-dev/incidents/spi-ticket#base-markup-required-fields" +
                    " базовые поля инцидента))");
            container.withNotification();
        } else {
            container.removeTag(TAG_BASE_MARKUP_REQUIRED);
            incident.removeTag(TAG_BASE_MARKUP_REQUIRED);
        }
    }

    private void inviteDutyToMarkUp(Incident incident, UpdateContainer container) {
        if (incident.hasTag(TAG_ADDED_HELP_LINK)
                || "neo".equals(incident.getStatus())
                || "diagnostics".equals(incident.getStatus())
        ) {
            return;
        }

        String responsible;
        if (!incident.getDuty().isEmpty()) {
            responsible = incident.getDuty().get(0);
        } else if (incident.getAssignee() != null) {
            responsible = incident.getAssignee();
        } else {
            responsible = incident.getAuthor();
        }
        if (responsible != null && responsible.startsWith("robot-")) {
            responsible = null;
        }

        String link = incident.isSPI() ? "https://docs.yandex-team.ru/direct-dev/incidents/spi-ticket#after-incident"
                : "https://docs.yandex-team.ru/direct-dev/incidents/ticket-markup-howto";
        String fieldLink = "https://docs.yandex-team.ru/direct-dev/incidents/spi-ticket#enable-fields";

        container.addTag(TAG_ADDED_HELP_LINK);
        incident.addTag(TAG_ADDED_HELP_LINK);
        if (responsible != null) {
            container.inviteToComment(responsible);
            container.addComment("кто:" + responsible + ", напиши пожалуйста:"
                    + "\n  - как авария проявила себя — в **Что во время аварии видели пользователи**"
                    + "\n  - когда и с чем сломалось, во сколько и как заметили, когда починили — в **Хронологию**"
                    + "\n  - свои действия — в раздел **Действия дежурного**"
                    + "\n  - приложи ссылки на логи, графики и прочее в раздел **Диагностика**"
                    + "\n  - в чем суть поломки — в **Описание** (или попроси заполнить более понимающего коллегу)");
        }
        container.addComment("Нужна помощь или ничего не понятно? Напиши нам в " + SLACK_LINK);
        container.addComment("В оформлении поможет наша документация — **(("
                + link + " Что заполнять после аварии))**" +
                ", ((" + fieldLink + " какие поля трекера включить))");
    }

    private String makeWhatToFillCommentAndTags(Incident incident, UpdateContainer container) {
        StringBuilder text = new StringBuilder();
        text.append("!!(green)Что нужно заполнить в этой аварии:!!");
        text.append(" !!(gray)//(комментарий обновляется автоматически)//!!\n");
        text.append("========(markup)\n"); // якорь для type-in scroll

        List<String> list = new ArrayList<>();
        Map<String, String> parsed = parseSpiDescription(incident.getDescription());
        if (parsed.getOrDefault(DESC_WHAT_SEEN, "").length() < 10) {
            list.add(DESC_WHAT_SEEN);
            container.addTagIfNeeded(TAG_REQUIRED_WHAT_SEEN, incident);
        } else {
            container.removeTagIfNeeded(TAG_REQUIRED_WHAT_SEEN, incident);
        }
        if (parsed.getOrDefault(DESC_DESC, "").length() < 30) {
            list.add(DESC_DESC);
            container.addTagIfNeeded(TAG_REQUIRED_DESCRIPTION, incident);
        } else {
            container.removeTagIfNeeded(TAG_REQUIRED_DESCRIPTION, incident);
        }
        if (parsed.getOrDefault(DESC_CHRONOLOGY, "").length() < 60) {
            list.add(DESC_CHRONOLOGY);
            container.addTagIfNeeded(TAG_REQUIRED_CHRONOLOGY, incident);
        } else {
            container.removeTagIfNeeded(TAG_REQUIRED_CHRONOLOGY, incident);
        }
        if (incident.getBeginTimeInstant() == null) {
            list.add("((https://docs.yandex-team.ru/direct-dev/incidents/spi-ticket#begin-time SRE: Время начала))");
        }
        if (incident.getEndTimeInstant() == null) {
            list.add("((https://docs.yandex-team.ru/direct-dev/incidents/spi-ticket#end-time SRE: Время окончания))");
        }
        if (incident.getImpact() == null) {
            list.add("((https://docs.yandex-team.ru/direct-dev/incidents/spi-ticket#impact SRE: impact))");
        }
        if (isBlank(incident.getSource())) {
            list.add("((https://docs.yandex-team.ru/direct-dev/incidents/spi-ticket#source " +
                    "Рекрутмент: Описание источника))");
        }
        if (parsed.getOrDefault(DESC_DUTY_ACTIONS, "").length() < 10) {
            list.add(DESC_DUTY_ACTIONS);
        }
        if (parsed.getOrDefault(DESC_DIAGNOSTIC, "").length() < 10) {
            list.add(DESC_DIAGNOSTIC);
        }

        if (incident.getImpactClients() == null) {
            list.add("((https://docs.yandex-team.ru/direct-dev/incidents/spi-ticket#impact-clients " +
                    "Влияние на клиентов))");
        }
        if (incident.getImpactTeam() == null) {
            list.add("((https://docs.yandex-team.ru/direct-dev/incidents/spi-ticket#impact-team Влияние на команду))");
        }

        text.append("**Дежурному app-duty**");
        addToDoList(text, list);
        list.clear();
        text.append("При необходимости добавь теги: ");
        text.append("((https://docs.yandex-team.ru/direct-dev/incidents/spi-ticket#external-service " +
                "про внешние сервисы)), ");
        text.append("((https://docs.yandex-team.ru/direct-dev/incidents/spi-ticket#frontend-bug " +
                "баг фронта)) %%direct:@direct_front%%, ");
        text.append("((https://docs.yandex-team.ru/direct-dev/incidents/spi-ticket#rollback откат)) %%rollback%% или ");
        text.append("((https://docs.yandex-team.ru/direct-dev/incidents/spi-ticket#money деньги)) %%direct:$$$%%.");
        text.append('\n');

        if (parsed.getOrDefault(DESC_PREVENTION_MEASURES, "").length() < 10) {
            list.add(DESC_PREVENTION_MEASURES);
        }
        if (incident.getSupportLine() == null) {
            list.add("((https://docs.yandex-team.ru/direct-dev/incidents/spi-ticket#support-line SRE: support_line))");
        }
        if (incident.getSreArea().isEmpty()) {
            list.add("((https://docs.yandex-team.ru/direct-dev/incidents/spi-ticket#sre-area SRE: SRE area))");
        }
        String losses = parsed.getOrDefault(DESC_LOSSES, "").toLowerCase();
        if (losses.length() < 10) {
            list.add(DESC_LOSSES);
        }
        if (incident.getVdt() == null) {
            list.add("((https://docs.yandex-team.ru/direct-dev/incidents/spi-ticket#vdt SRE: Даунтайм вертикали))");
        }
        if (incident.getYdt() == null) {
            list.add("((https://docs.yandex-team.ru/direct-dev/incidents/spi-ticket#ydt SRE: Даунтайм Яндекса))");
        }
        text.append("**Разбирающему аварию**");
        addToDoList(text, list);
        list.clear();
        text.append("Проверь значения SRE: ");
        text.append("((https://docs.yandex-team.ru/direct-dev/incidents/spi-ticket#notification-time " +
                "Время уведомления)), а также полей SRO: ");
        text.append("((https://docs.yandex-team.ru/direct-dev/incidents/spi-ticket#incident-reporter " +
                "Incident Reporter)), ");
        text.append("((https://docs.yandex-team.ru/direct-dev/incidents/spi-ticket#incident-protocol " +
                "Incident Protocol)), ");
        text.append("((https://docs.yandex-team.ru/direct-dev/incidents/spi-ticket#minus-dc Минус 1 ДЦ )).");

        text.append("\n\n<{!!(gray)Вся разметка Директа в сборе:!!\n");
        text.append("Хронология:");
        text.append("\n- ").append(formatTime(incident.getBeginTimeInstant())).append(" — begin");
        text.append("\n- ").append(formatTime(incident.getDetectTimeInstant())).append(" — notification");
        text.append("\n- ").append(formatTime(incident.getEndTimeInstant())).append(" — end");
        text.append("\nИсточник обнаружения: ");
        text.append(nvl(incident.getSource(), "не заполнен"));
        text.append("\nimpact: ");
        text.append(nvl(incident.getImpact(), "не заполнен"));
        text.append("\nКомпоненты:");
        incident.getComponents()
                .stream()
                .sorted()
                .map(c -> "\n- " + c)
                .forEach(text::append);
        text.append("\nТеги:");
        incident.getTags()
                .stream()
                .sorted()
                .map(c -> "\n- " + c)
                .forEach(text::append);
        text.append("}>");

        return text.toString();
    }

    private void addToDoList(StringBuilder text, List<String> list) {
        if (list.isEmpty()) {
            text.append(" — !!(green)всё заполнено!!\n");
        } else {
            text.append(':');
            text.append('\n');
            for (String el : list) {
                text.append("- ");
                text.append(el);
                text.append('\n');
            }
        }
    }


    /**
     * Для удобства подсчета статистики: примерно транслируем понимание "solved" инцидента из Warden
     * в значение поля "Разбор ошибок" — по которому удобно считать статистку
     * <p>
     * Исходник реализации расчета Solved:
     * https://a.yandex-team.ru/arc/trunk/arcadia/search/mon/warden/src/workers/startrek_sync.py?rev=7596272#L256
     */
    private void processSolvedSpi(Incident incident, UpdateContainer container) {
        if (!incident.isSPI()) {
            return;
        }

        String newFiguredOut = isSpiSolved(incident);
        if (!Objects.equals(newFiguredOut, incident.getFiguredOut())) {
            logger.info("Current figuredOut for {} is {}, assumed — {}. Let's update",
                    incident.getKey(), incident.getFiguredOut(), newFiguredOut);
            container.set(FIGURED_OUT_FIELD_NAME, newFiguredOut);
            incident.setFiguredOut(newFiguredOut);
        }
    }

    private void fixSpiTitleFromTags(Incident incident, UpdateContainer container) {
        if (!incident.isSPI()) {
            return;
        }
        String current = incident.getSpiSummary();
        String desired = incident.composeSpiSummary();
        if (current == null || desired == null || current.equals(desired)) {
            return;
        }

        logger.info("Summary for {} is {}, assumed — {}. Let's update", incident.getKey(), current, desired);
        container.summary(desired);
        incident.setSummary(desired);
    }

    private void fillIncidentReporterFromSource(Incident incident, UpdateContainer container) {
        String current = incident.getIncidentReporter();
        if (!incident.isSPI()
                || isBlank(incident.getSource())
                || current != null && !current.equals(Incident.USER_REPORTER)
        ) {
            return;
        }

        String desired;
        switch (incident.getSource()) {
            case "жалобы пользователей":
                desired = incident.hasComponent("#hype") ? "external" : Incident.USER_REPORTER;
                break;
            case Incident.MONITORING_SOURCE:
                desired = Incident.MONITORING_REPORTER;
                break;
            case "сигнал от смежников":
                desired = Incident.USER_REPORTER;
                break;
            case Incident.SERVICE_SOURCE:
                desired = Incident.SERVICE_REPORTER;
                break;
            default:
                return;
        }

        if (desired.equals(current)) {
            return;
        }

        logger.info("Incident Reporter for {} is {}, assumed — {}. Let's update", incident.getKey(), current, desired);
        container.set(INCIDENT_REPORTER_FIELD_NAME, desired);
        incident.setIncidentReporter(desired);
    }

    private void fillSourceFromIncidentReporter(Incident incident, UpdateContainer container) {
        String reporter = incident.getIncidentReporter();

        if (!incident.isSPI()
                || reporter == null
                || isNotBlank(incident.getSource())
        ) {
            return;
        }

        String source;
        switch (reporter) {
            case Incident.SERVICE_REPORTER:
                source = Incident.SERVICE_SOURCE;
                break;
            case Incident.MONITORING_REPORTER:
                source = Incident.MONITORING_SOURCE;
                break;
            default:
                return;
        }

        logger.info("Guessed {} for {} is {}", SOURCE_DESCRIPTION_FIELD_NAME, incident.getKey(), source);
        container.set(SOURCE_DESCRIPTION_FIELD_NAME, source);
        incident.setSource(source);
    }

    private void setZeroClientsImpactForInternalIncidents(Incident incident, UpdateContainer container) {
        if (incident.getImpact() == null || !incident.getImpact().equals("internal")) {
            return;
        }

        boolean hasImpactClients = incident.getComponents()
                .stream()
                .anyMatch(Incident.IMPACT_CLIENTS::containsKey);
        if (hasImpactClients) {
            return;
        }

        logger.info("Guessed {} for {} from impact=internal", IMPACT_CLIENTS_0, incident.getKey());
        if (incident.isSPI()) {
            String tag = spiTag(IMPACT_CLIENTS_0);
            container.addTag(tag);
            incident.addTag(tag);
        } else {
            Component component = componentsLoader.getUnchecked(IMPACT_CLIENTS_0);
            container.addComponent(component);
            incident.addComponent(component.getDisplay());
        }
    }

    @Nonnull
    private String isSpiSolved(Incident incident) {
        final String solved = "Да";
        final String notSolved = "Нет";

        logger.debug("Checking that {} is solved", incident.getKey());
        if (incident.hasTag(StartrekTag.SPI_VICTIM)) {
            return solved;
        }

        List<String> links = client.links().getLocal(incident.getKey())
                .stream()
                .map(LocalLink::getObject)
                .map(IssueRef::getKey)
                .collect(Collectors.toList());

        if (incident.getStatus().equals("inDevelopment")) {
            for (String link : links) {
                if (isActionItem(link)) {
                    return solved;
                }
            }
            return notSolved;
        }

        if (!incident.getStatus().equals("closed")) {
            return notSolved;
        }

        String resolution = incident.getResolution();
        if ("can'tReproduce".equals(resolution)) {
            return solved;
        } else if ("duplicate".equals(resolution) || "relapse".equals(resolution)) {
            if (links.stream().anyMatch(l -> l.startsWith(StartrekQueue.SPI + "-"))) {
                return solved;
            }
        } else if ("fixed".equals(resolution)) {
            for (String link : links) {
                if (link.startsWith(StartrekQueue.SPPROBLEM + "-")) {
                    return solved;
                } else if (isActionItem(link)) {
                    return solved;
                }
            }
        }

        return notSolved;
    }

    /**
     * Проверить, стоит ли на тикете тег spi:actionitem
     */
    private boolean isActionItem(String id) {
        if (id.startsWith(StartrekQueue.SPI + "-")) {
            return false;
        }
        if (id.startsWith(StartrekQueue.SPPROBLEM + "-")) {
            return false;
        }
        try {
            return client.issues().get(id)
                    .getTags()
                    .stream()
                    .anyMatch(StartrekTag.SPI_ACTION_ITEM::equals);
        } catch (ForbiddenException ignored) {
            logger.error("Failed to load related issue {} - no access", id);
            return false;
        }
    }

    private void saveProgress(LocalDate maxUpdatedAt) {
        LocalDate yesterday = LocalDate.now().minusDays(1);
        if (maxUpdatedAt == null || yesterday.compareTo(maxUpdatedAt) > 0) {
            logger.info("there is no fresh tickets, setting yesterday as last date");
            maxUpdatedAt = yesterday;
        }
        settings.setSetting(APP, SETTING_MAX_UPDATED_AT, ISO_LOCAL_DATE.format(maxUpdatedAt));
    }

    static class UpdateContainer extends IssueUpdate.Builder {

        private final List<String> tagsToAdd;
        private final List<String> tagsToRemove;
        private final List<String> comments;
        private final List<Component> componentsToAdd;
        private final List<Component> componentToRemove;
        private boolean notify;
        private String invite;

        UpdateContainer() {
            super();
            tagsToAdd = new ArrayList<>();
            tagsToRemove = new ArrayList<>();
            componentsToAdd = new ArrayList<>();
            componentToRemove = new ArrayList<>();
            comments = new ArrayList<>();
            notify = false;
            invite = null;
        }

        public IssueUpdate build() {
            if (!tagsToAdd.isEmpty() || !tagsToRemove.isEmpty()) {
                super.tags(Cf.wrap(tagsToAdd), Cf.wrap(tagsToRemove));
            }
            if (!componentsToAdd.isEmpty() || !componentToRemove.isEmpty()) {
                super.components(Cf.wrap(componentsToAdd), Cf.wrap(componentToRemove));
            }
            if (!comments.isEmpty()) {
                String prefix = comments.size() == 1 ? "" : "* ";
                var comment = signedMessageFactory(this, SELF, prefix + String.join("\n* ", comments));
                if (invite != null) {
                    comment = comment.summonees(invite);
                }
                super.comment(comment.build());
            }

            return super.build();
        }

        IncidentUpdate buildIncidentUpdate() {
            return new IncidentUpdate(this.build(), notify);
        }

        void addTagIfNeeded(String tag, Incident incident) {
            if (incident.hasTag(tag)) {
                return;
            }
            addTag(tag);
        }

        void removeTagIfNeeded(String tag, Incident incident) {
            if (!incident.hasTag(tag)) {
                return;
            }
            removeTag(tag);
        }

        void addTag(String tag) {
            tagsToAdd.add(tag);
        }

        void removeTag(String tag) {
            tagsToRemove.add(tag);
        }

        void addComponent(Component component) {
            componentsToAdd.add(component);
        }

        void removeComponent(Component component) {
            componentToRemove.add(component);
        }

        void addComment(String comment) {
            comments.add(comment);
        }

        void withNotification() {
            notify = true;
        }

        void inviteToComment(String login) {
            invite = login;
        }
    }

    static class IncidentUpdate {
        private final boolean notify;
        private final IssueUpdate update;

        IncidentUpdate(IssueUpdate update, boolean notify) {
            this.update = update;
            this.notify = notify;
        }

        public boolean isNotify() {
            return notify;
        }

        public IssueUpdate getUpdate() {
            return update;
        }

        public boolean isEmpty() {
            return getUpdate().getValues().isEmpty();
        }
    }

    static String spiTag(String tag) {
        return StartrekTag.SPI_PREFIX + tag;
    }
}
