package ru.yandex.direct.chassis.entity.reports.incident;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Random;
import java.util.TreeMap;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.Function;
import java.util.stream.Collectors;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.stereotype.Component;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.startrek.client.Session;
import ru.yandex.startrek.client.model.CommentCreate;
import ru.yandex.startrek.client.model.UserRef;

import static ru.yandex.direct.chassis.configuration.StartrekConfigurationKt.MAINTENANCE_HELPERS_TRACKER_BEAN;
import static ru.yandex.direct.chassis.entity.reports.incident.IncidentsReporter.SLACK_LINK;

/**
 * Тул для призыва в комментарии по инцидентам, в которых нужно проверить разметку.
 * YQL для тикетов https://yql.yandex-team.ru/Operations/X092_yAsJQ_sa45WTVTYwPCg_y9MvGkY__VT7NlABQM=
 */
@Component
public class IncidentsReviewHelper implements Runnable {
    private static final Logger logger = LoggerFactory.getLogger(IncidentsReviewHelper.class);

    private final Session client;
    private IncidentsReviewHelper(@Qualifier(MAINTENANCE_HELPERS_TRACKER_BEAN) Session session) {
        this.client = session;
    }

    @Override
    public void run() {
        //------------------- заполни этот блок -----------------
        // тикет "смотр", дочерний к "Direct Incident Review <date>"
        boolean removeInviter = false;  // убирать ли из призыва того, кто запускает призыв
        String result = "DIRECT-NNNN";
        List<String> todo = Arrays.asList(
        );
        var reviewers = List.of("yukaba", "maxlog", "ppalex", "mokeev", "bratgrim", "gerdler", "pe4kin", "andreypav", "darkkeks");
        //-------------- конец редактируемого блока -------------

        String currentUser = System.getenv("USER");
        String parent = result;

        Random random = new Random(result.hashCode());
        Collections.shuffle(todo, random);

        Map<String, CommentCreate> comments = new TreeMap<>();
        Map<String, List<String>> history = new HashMap<>();
        Map<String, AtomicInteger> counter = new HashMap<>();
        reviewers.forEach(login -> counter.put(login, new AtomicInteger()));
        Comparator<String> comparator = Comparator.comparingInt(login -> counter.get(login).get());

        for (var incident : todo) {
            // разметившие
            List<String> marked = client.issues().get(incident)
                    .<List<UserRef>>getO("marked").orElse(List.of())
                    .stream()
                    .map(UserRef::getLogin)
                    .collect(Collectors.toList());
            if (removeInviter) {
                marked.add(currentUser);
            }

            ArrayList<String> copy = new ArrayList<>(reviewers);
            marked.forEach(copy::remove);
            Collections.shuffle(copy, random);
            copy.sort(comparator);

            List<String> summonees = copy.subList(0, marked.size() < 2 ? 2 : 1);
            summonees.forEach(login -> counter.get(login).incrementAndGet());
            history.put(incident, summonees);


            String who = summonees.stream()
                    .map(login -> "кто:" + login)
                    .collect(Collectors.joining(", "));

            String instructions = incident.startsWith("SPI-")
                    ? "https://docs.yandex-team.ru/direct-dev/incidents/spi-ticket#after-incident"
                    : "https://docs.yandex-team.ru/direct-dev/incidents/ticket-markup-howto";

            CommentCreate comment = CommentCreate.builder()
                    .summonees(Cf.wrap(summonees))
                    .comment(who + " — пожалуйста, " +
                            (summonees.size() == 1 ? "проверь " : "проверьте ") +
                            "правильность разметки этого инцидента:\n" +
                            "- в описании и \"что видел пользователь\" — написано что-то понятное\n" +
                            "- время начала, уведомления и завершения\n" +
                            "- записана текстом хронология событий\n" +
                            "- влияние на команду и клиентов\n" +
                            "- источник обнаружения\n" +
                            "- другие компоненты или теги по ((" + instructions + " инструкции))\n" +
                            "Для удобства вся разметка собрана " +
                            "((https://st.yandex-team.ru/" + incident + "#markup под катом))\n" +
                            "\nЕсли разметка ОК – поставь лайк этому комментарию.\n\n" +
                            "Если есть вопрос или замечание — спроси комментарием.\n" +
                            "Сомневаешься или ничего не понятно — пиши нам в " + SLACK_LINK + "\n" +
                            "Успеть желательно ко встрече по разбору (обычно по понедельникам в 18 часов).")
                    .build();
            comments.put(incident, comment);
        }

        logger.info("who is called: {}", history);
        logger.info("stat by logins: {}", counter);

        Function<String, String> formatIncident = (incident) -> String.format("- %s — проверяют: %s\n",
                incident,
                String.join(", ", history.get(incident))
        );

        String listResult = StreamEx.of(todo).sorted().map(formatIncident).collect(Collectors.joining());
        var summary = new StringBuilder()
                .append("====Тикеты\n")
                .append(listResult)
                .append("\n\n====Проверяющие\n");

        EntryStream.of(history)
                .flatMapValues(Collection::stream)
                .invert()
                .sortedBy(Map.Entry::getKey)
                .collapseKeys()
                .forKeyValue((l, i) ->
                        summary.append("\nкто:")
                                .append(l)
                                .append(":\n- ")
                                .append(String.join("\n- ", i))
                                .append("\n")
                );

        CommentCreate list = CommentCreate.builder().comment(summary.toString()).build();
        comments.put(result, list);

        logger.info("Post comments");
        comments.forEach((t, c) -> client.comments().create(t, c, false, false));
    }
}
