package ru.yandex.direct.chassis.entity.reports.incident;

import java.time.Instant;
import java.time.LocalDateTime;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.yandex.ydb.core.Result;
import com.yandex.ydb.table.description.TableDescription;
import com.yandex.ydb.table.query.Params;
import com.yandex.ydb.table.values.PrimitiveType;
import com.yandex.ydb.table.values.PrimitiveValue;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.chassis.util.startrek.StartrekQueue;
import ru.yandex.direct.chassis.util.ydb.YdbClient;
import ru.yandex.direct.liveresource.LiveResourceFactory;

import static com.yandex.ydb.table.transaction.TxControl.serializableRw;
import static ru.yandex.direct.chassis.util.ydb.YdbClient.KEEP_IN_CACHE;
import static ru.yandex.direct.utils.DateTimeUtils.moscowDateTimeToInstant;

@Component
@ParametersAreNonnullByDefault
public class IncidentsScoring {
    private static final Logger logger = LoggerFactory.getLogger(IncidentsScoring.class);

    private static final String TO_MARK_UP_QUERY_SOURCE = "classpath:///scoring_of_incidents_to_mark_up.yql";
    private static final String TO_FIGURE_OUT_QUERY_SOURCE = "classpath:///scoring_of_incidents_to_figure_out.yql";
    private static final String INCIDENTS_STAT_TEMPLATE_QUERY_SOURCE = "classpath:///incidents_stat_template.yql";
    private static final TableDescription TO_MARK_UP_TABLE = TableDescription.newBuilder()
            .addNullableColumn("scoreTime", PrimitiveType.datetime())
            .addNullableColumn("count", PrimitiveType.uint64())
            .addNullableColumn("score", PrimitiveType.float64())
            .addNullableColumn("debt", PrimitiveType.float64())
            .addNullableColumn("relevanceTime", PrimitiveType.datetime())
            .addNullableColumn("reviewCount", PrimitiveType.uint64())
            .addNullableColumn("reviewScore", PrimitiveType.float64())
            .setPrimaryKey("scoreTime")
            .build();
    private static final TableDescription TO_FIGURE_OUT_TABLE = TableDescription.newBuilder()
            .addNullableColumn("scoreTime", PrimitiveType.datetime())
            .addNullableColumn("count", PrimitiveType.uint64())
            .addNullableColumn("relevanceTime", PrimitiveType.datetime())
            .setPrimaryKey("scoreTime")
            .build();
    private static final TableDescription INCIDENTS_STAT = TableDescription.newBuilder()
            .addNullableColumn("report", PrimitiveType.utf8())
            .addNullableColumn("reportTime", PrimitiveType.datetime())
            .addNullableColumn("solvedIncidents", PrimitiveType.uint64())
            .addNullableColumn("notSolvedIncidents", PrimitiveType.uint64())
            .addNullableColumn("closedActionItems", PrimitiveType.uint64())
            .addNullableColumn("openActionItems", PrimitiveType.uint64())
            .setPrimaryKeys("report", "reportTime")
            .build();
    private static final String SPI_CONDITION = "key LIKE '" + StartrekQueue.SPI + "-%'";
    private static final String DETECTED_AT_CONDITION = "detectTime BETWEEN $beginTime AND $endTime";
    private static final String DIRECT_CONDITION = "service = 'direct'";
    private static final LocalDateTime Q2_2021 = LocalDateTime.of(2021, 4, 1, 0, 0);
    private static final LocalDateTime Q4_2021 = LocalDateTime.of(2021, 10, 1, 0, 0);
    private static final LocalDateTime Q2_2022 = LocalDateTime.of(2022, 4, 1, 0, 0);
    private static final LocalDateTime Q4_2022 = LocalDateTime.of(2022, 10, 1, 0, 0);

    private final YdbClient ydb;
    private final String toMarkUpQuery;
    private final String toFigureOutQuery;
    private final String statQueryTemplate;

    @Autowired
    public IncidentsScoring(YdbClient ydb) {
        this.ydb = ydb;
        toMarkUpQuery = LiveResourceFactory.get(TO_MARK_UP_QUERY_SOURCE).getContent();
        toFigureOutQuery = LiveResourceFactory.get(TO_FIGURE_OUT_QUERY_SOURCE).getContent();
        statQueryTemplate = LiveResourceFactory.get(INCIDENTS_STAT_TEMPLATE_QUERY_SOURCE).getContent();
    }

    /**
     * "Очки" неразмеченных тикетов.
     * <p>
     * Текущая формула - за единицу принимаем 1 тикет в котором нужно разметить 8 полей.
     * Долг начисляется - 1 за каждую неделю с третьего дня создания тикета.
     */
    public void scoreIncidentsToMarkUp() {
        Instant now = Instant.now();
        logger.info("calculate and write score of incidents to mark up at {}", now);
        Params params = Params.of("$now", PrimitiveValue.datetime(now),
                "$border_days", PrimitiveValue.float64(3)
        );
        ydb.supplyStatus(s -> s.executeDataQuery(toMarkUpQuery, serializableRw(), params, KEEP_IN_CACHE)
                .thenApply(Result::toStatus))
                .expect("error calculating score of incidents to mark up");
    }

    public void scoreIncidentsToFigureOut() {
        Instant now = Instant.now();
        logger.info("calculate and write score of incidents to figure out at {}", now);
        Params params = Params.of(
                "$now", PrimitiveValue.datetime(now)
        );
        ydb.supplyStatus(s -> s.executeDataQuery(toFigureOutQuery, serializableRw(), params, KEEP_IN_CACHE)
                .thenApply(Result::toStatus))
                .expect("error calculating score of incidents to figure up");
    }

    public void directStat2021Q2() {
        incidentsStat("2021/Q2 - 2021/Q3 direct", IncidentsConstants.BY_DETECT_TIME_IDX,
                DIRECT_CONDITION + " AND " + DETECTED_AT_CONDITION,
                Q2_2021, Q4_2021);
    }

    public void directStat2021Q4() {
        incidentsStat("2021/Q4 - 2022/Q1 direct", IncidentsConstants.BY_DETECT_TIME_IDX,
                DIRECT_CONDITION + " AND " + DETECTED_AT_CONDITION,
                Q4_2021, Q2_2022);
    }

    public void directStat2022Q2() {
        incidentsStat("2022/Q2 - 2022/Q3 direct", IncidentsConstants.BY_DETECT_TIME_IDX,
                DIRECT_CONDITION + " AND " + DETECTED_AT_CONDITION,
                Q2_2022, Q4_2022);
    }

    public void allStat() {
        incidentsStat("All", IncidentsConstants.BY_SERVICE_IDX, DIRECT_CONDITION, null, null);
    }

    public void spiStat() {
        incidentsStat("SPI", IncidentsConstants.BY_SERVICE_IDX, SPI_CONDITION + " AND " + DIRECT_CONDITION, null, null);
    }

    public void createTables() {
        ydb.createTable("scoring_of_incidents_to_mark_up", TO_MARK_UP_TABLE);
        ydb.createTable("scoring_of_incidents_to_figure_out", TO_FIGURE_OUT_TABLE);
        ydb.createTable("incidents_stat", INCIDENTS_STAT);
    }

    private void incidentsStat(String report, String view, String where,
                               @Nullable LocalDateTime from, @Nullable LocalDateTime to) {
        if (!view.isBlank()) {
            view = "VIEW " + view;
        }
        if (!where.isBlank()) {
            where = "WHERE " + where;
        }
        String query = String.format(statQueryTemplate, report, view, where);
        Instant now = Instant.now();

        logger.info("process report '{}' for incidents and action items filtered by: {} ({} - {})",
                report, where, from, to);

        Params params = Params.create().put("$now", PrimitiveValue.datetime(now));
        if (from != null) {
            params.put("$beginTime", PrimitiveValue.datetime(moscowDateTimeToInstant(from)));
        }
        if (to != null) {
            params.put("$endTime", PrimitiveValue.datetime(moscowDateTimeToInstant(to)));
        }

        ydb.supplyStatus(s -> s.executeDataQuery(query, serializableRw(), params, KEEP_IN_CACHE)
                .thenApply(Result::toStatus))
                .expect("error calculating incidents statistics");
    }
}
