package ru.yandex.direct.chassis.entity.sandbox

import io.netty.handler.codec.http.HttpHeaders
import org.asynchttpclient.AsyncHttpClient
import org.slf4j.LoggerFactory
import org.springframework.beans.factory.annotation.Value
import org.springframework.http.MediaType
import org.springframework.stereotype.Component
import ru.yandex.direct.asynchttp.FetcherSettings
import ru.yandex.direct.asynchttp.ParallelFetcherFactory
import ru.yandex.direct.http.smart.annotations.Json
import ru.yandex.direct.http.smart.annotations.ResponseHandler
import ru.yandex.direct.http.smart.core.Call
import ru.yandex.direct.http.smart.core.Smart
import ru.yandex.direct.http.smart.error.ErrorUtils
import ru.yandex.direct.http.smart.http.Body
import ru.yandex.direct.http.smart.http.GET
import ru.yandex.direct.http.smart.http.POST
import ru.yandex.direct.http.smart.http.Query
import ru.yandex.direct.liveresource.LiveResource
import ru.yandex.direct.liveresource.provider.LiveResourceFactoryBean

class SandboxException(message: String) : RuntimeException(message)

@Component
class SandboxClient(
    @Value("\${sandbox.token}") tokenPath: String,
    liveResourceFactoryBean: LiveResourceFactoryBean,
    asyncHttpClient: AsyncHttpClient,
) {
    private val fetcherFactory = ParallelFetcherFactory(asyncHttpClient, FetcherSettings())
    private val api = createApi(liveResourceFactoryBean.get(tokenPath), fetcherFactory)

    companion object {
        private const val SANDBOX_URL = "https://sandbox.yandex-team.ru/api/v1.0/"
        private val logger = LoggerFactory.getLogger(SandboxClient::class.java)
    }

    private fun createApi(oauthTokenResource: LiveResource, parallelFetcherFactory: ParallelFetcherFactory): Api {
        return Smart.builder()
            .withParallelFetcherFactory(parallelFetcherFactory)
            .withProfileName("sandbox_client")
            .withBaseUrl(SANDBOX_URL)
            .addHeaderConfigurator { headers: HttpHeaders ->
                headers.add(org.springframework.http.HttpHeaders.CONTENT_TYPE,
                    MediaType.APPLICATION_JSON_VALUE)
            }
            .addHeaderConfigurator { headers: HttpHeaders ->
                headers.add(org.springframework.http.HttpHeaders.AUTHORIZATION,
                    "OAuth ${oauthTokenResource.content.removeSuffix("\n")}")
            }
            .build()
            .create(Api::class.java)
    }

    interface Api {
        @POST("/release")
        @Json
        @ResponseHandler(expectedCodes = [202, 400])
        fun releaseTask(@Body @Json releaseRequest: ReleaseRequest): Call<ReleaseTaskResponse>

        @GET("/release")
        @Json
        @ResponseHandler(expectedCodes = [200])
        fun getRelease(@Query("task_id") id: Long, @Query("limit") limit: Int, @Query("type") releaseSubject: ReleaseSubject)
            : Call<ReleasesInfoItems>

    }


    fun getRelease(taskId: Long, subject: ReleaseSubject): ReleaseInfo? {
        val releaseInfoResult = api.getRelease(taskId, 1, subject).execute()

        ErrorUtils.checkResultForErrors(releaseInfoResult, { SandboxException(it) })
        return if (releaseInfoResult.success.items.isEmpty()) null else releaseInfoResult.success.items[0]
    }

    fun releaseTask(releaseRequest: ReleaseRequest) {
        val result = api.releaseTask(releaseRequest).execute()
        ErrorUtils.checkResultForErrors(result, { SandboxException(it) })
        if (result.success.reason != null) {
            logger.warn("Releasing finish with code 400: $result.success.reason")
        }
    }
}
