package ru.yandex.direct.chassis.entity.startrek

import ru.yandex.direct.chassis.util.DirectAppsConfEntry
import ru.yandex.direct.chassis.util.DirectAppsConfProvider
import ru.yandex.direct.chassis.util.DirectReleaseUtils
import ru.yandex.direct.chassis.util.startrek.StartrekHelper
import ru.yandex.direct.chassis.util.nonCf
import ru.yandex.direct.env.ProductionOnly
import ru.yandex.direct.scheduler.Hourglass
import ru.yandex.direct.scheduler.support.DirectJob
import ru.yandex.startrek.client.model.Issue

const val AUTOCOMMENTS_TAG = "autocomments"

const val UNIT_TESTS_ITEM = "Юнит-тесты"
const val AUTO_TESTS_ITEM = "Автотесты"
const val MANUAL_TESTING_ITEM = "Тикеты \"Тестировать\""
const val SORT_OUT_ITEM = "Тикеты \"Разобраться\""
const val DEV_ENVIRONMENT_ITEM = "devtest/dev7"
const val RELEASE_MANUAL_HEALTH_CHECK = "Проверка работоспособности релиза (см. https://nda.ya.ru/t/bVfqq3QF5Ddbbp )"

val CHECKLIST_ITEM_NAMES = listOfNotNull(
    UNIT_TESTS_ITEM,
    AUTO_TESTS_ITEM,
    MANUAL_TESTING_ITEM,
    SORT_OUT_ITEM,
    DEV_ENVIRONMENT_ITEM,
    RELEASE_MANUAL_HEALTH_CHECK,
)

const val DIRECT_APP = "direct"
const val JAVA_INTAPI_APP = "java-intapi"
const val JAVA_LOGVIEWER_APP = "java-logviewer"
const val JAVA_API5_APP = "java-api5"
const val JAVA_WEB_APP = "java-web"
const val JAVA_JOBS_APP = "java-jobs"
const val ESS_ROUTER_APP = "ess-router"
const val BINLOGBROKER_APP = "binlogbroker"
const val JAVA_ALW_APP = "java-alw"

val REGRESSION_APP_NAMES = setOf(
    DIRECT_APP,
    JAVA_INTAPI_APP,
    JAVA_API5_APP,
    JAVA_WEB_APP,
)

val RELEASES_WITH_MANUAL_HEALTH_CHECK = setOf(
    JAVA_JOBS_APP,
    ESS_ROUTER_APP,
    BINLOGBROKER_APP,
    JAVA_ALW_APP,
)

val NO_DEV_ENVIRONMENT_APP_NAMES = listOfNotNull(
    JAVA_LOGVIEWER_APP
)

@Hourglass(periodInSeconds = 5 * 60, needSchedule = ProductionOnly::class)
class AutocommentReleasesJob(
    val startrek: StartrekHelper,
    val directAppsConfProvider: DirectAppsConfProvider,
) : DirectJob() {
    override fun execute() {
        val query = """
            | Queue: DIRECT
            | Type: Release
            | Components: "Releases: JavaDirect"
            | Status: Testing, ReadyForTest
            | Tags: !"$AUTOCOMMENTS_TAG"
            | "Sort by": key desc
        """.trimMargin()

        val releases = startrek.session.issues().find(query).toList().nonCf()

        for (release in releases) {
            addStartrekChecklist(release)
        }
    }

    private val appByName: Map<String, DirectAppsConfEntry>
        get() = directAppsConfProvider.getDirectAppsConf()
            .associateBy { it.name }

    private fun addStartrekChecklist(release: Issue) {
        val app = DirectReleaseUtils.getReleaseApp(release, appByName.values)
        val checkListItemName = CHECKLIST_ITEM_NAMES
            .filter { it != AUTO_TESTS_ITEM || app?.name in REGRESSION_APP_NAMES }
            .filter { it != DEV_ENVIRONMENT_ITEM || app?.name !in NO_DEV_ENVIRONMENT_APP_NAMES }
            .filter { it != RELEASE_MANUAL_HEALTH_CHECK || app?.name in RELEASES_WITH_MANUAL_HEALTH_CHECK }
        startrek.addChecklist(release, checkListItemName, ifNoTag = AUTOCOMMENTS_TAG)
    }
}
