package ru.yandex.direct.chassis.entity.startrek

import org.slf4j.LoggerFactory
import org.tmatesoft.svn.core.SVNLogEntryPath
import org.tmatesoft.svn.core.SVNURL
import org.tmatesoft.svn.core.wc.SVNClientManager
import org.tmatesoft.svn.core.wc.SVNRevision
import ru.yandex.direct.chassis.util.DirectAppsConfProvider
import ru.yandex.direct.chassis.properties.PropertiesLightSupport
import ru.yandex.direct.chassis.util.startrek.StartrekHelper
import ru.yandex.direct.chassis.util.Utils
import ru.yandex.direct.chassis.util.github.GitHubClient
import ru.yandex.direct.chassis.util.startrek.NONE_AFFECTED_APP
import ru.yandex.direct.chassis.util.startrek.StartrekDisplayStatus
import ru.yandex.direct.chassis.util.startrek.StartrekField
import ru.yandex.direct.chassis.util.startrek.StartrekQueue
import ru.yandex.direct.chassis.util.startrek.StartrekStatusKey
import ru.yandex.direct.chassis.util.startrek.StartrekTag
import ru.yandex.direct.env.ProductionOnly
import ru.yandex.direct.scheduler.Hourglass
import ru.yandex.direct.scheduler.support.DirectJob
import ru.yandex.startrek.client.model.Issue
import java.text.SimpleDateFormat
import java.time.ZonedDateTime
import java.time.format.DateTimeFormatter

private const val WORKFLOW_GLITCH_TAG = "workflow_glitch"
private const val MAX_ISSUES_STARTREK_FILTER_COUNT = 70
private const val DOCUMENTATION_WORKFLOW_TEXT =
    "Подробнее о воркфлоу задач — https://docs.yandex-team.ru/direct-dev/concepts/dev/ticket-workflow"

private const val ARCADIA_REPOSITORY = "svn+ssh://arcadia.yandex.ru/arc/trunk/arcadia"
private const val DIRECT_REPOSITORY = "$ARCADIA_REPOSITORY/direct/"
private const val FRONTEND_REPOSITORY = "$ARCADIA_REPOSITORY/adv/frontend/"

private const val LAST_PROCESSED_REVISION_PROPERTY = "awaiting_release_last_processed_revision"
private const val AWAITING_RELEASE_PROCESSOR_ENABLED = "awaiting_release_processor_enabled"

private const val DATE_TIME_PATTERN = "yyyy-MM-dd'T'HH:mm:ssZ"

/**
 * Коммиты в эти директории не относятся ни к какому приложению
 */
private val FILTERED_PATHS = setOf(
    "/trunk/arcadia/adv/frontend/docs",
    "/trunk/arcadia/direct/apps/logbroker-reader",
    "/trunk/arcadia/direct/apps/maintenance-helpers",
    "/trunk/arcadia/direct/autotests",
    "/trunk/arcadia/direct/backend",
    "/trunk/arcadia/direct/docs",
    "/trunk/arcadia/direct/infra",
    "/trunk/arcadia/direct/packages",
    "/trunk/arcadia/direct/qa",
    "/trunk/arcadia/direct/schemata",
    "/trunk/arcadia/direct/solo",
)

/**
 * Коммит, после которого будут обрабатываться коммиты в аркадию в первом запуске
 */
private const val FIRST_LAUNCH_ARCADIA_REVISION = 8000000L

/**
 * Давность комммита, чтобы его можно было обработать
 */
private const val BACK_COMMITS_MINUTES = 15L

/**
 * Обработка статуса [AWAITING_RELEASE_DISPLAY_STATUS]
 *
 * См. подробное описание в [DIRECT-131678](https://st.yandex-team.ru/DIRECT-131678)
 */
@Hourglass(periodInSeconds = 15 * 60, needSchedule = ProductionOnly::class)
class AwaitingReleaseProcessor(
    private val startrek: StartrekHelper,
    private val gitHubClient: GitHubClient,
    private val svnClientManager: SVNClientManager,
    private val propertiesLightSupport: PropertiesLightSupport,
    private val directAppsConfProvider: DirectAppsConfProvider,
) : DirectJob() {

    companion object {
        private val logger = LoggerFactory.getLogger(AwaitingReleaseProcessor::class.java)
        private val VALID_STATUSES_FOR_AWAITING_RELEASE_MOVE = setOf(StartrekStatusKey.BETA_TESTED)
        private val VALID_STATUSES_AFTER_COMMIT = setOf(StartrekStatusKey.BETA_TESTED, StartrekStatusKey.CLOSED)
        private val DATE_TIME_FORMATTER = DateTimeFormatter.ofPattern(DATE_TIME_PATTERN)
    }

    private val isProcessorEnabled: Boolean
        get() {
            // По умолчанию джоба включена
            val value = propertiesLightSupport[AWAITING_RELEASE_PROCESSOR_ENABLED] ?: return true
            return try {
                value.toInt() == 1
            } catch (e: NumberFormatException) {
                logger.error("Failed to parse $AWAITING_RELEASE_PROCESSOR_ENABLED = $value")
                false
            }
        }

    override fun execute() {
        if (!isProcessorEnabled) {
            logger.info("Skip processing. AwaitingReleaseProcessor is not enabled")
            return
        }

        val lastProcessedRevision = propertiesLightSupport[LAST_PROCESSED_REVISION_PROPERTY]?.toLong()
            ?: FIRST_LAUNCH_ARCADIA_REVISION
        val now = DATE_TIME_FORMATTER.format(ZonedDateTime.now().minusMinutes(BACK_COMMITS_MINUTES))

        val (startRevision, endRevision) = getRevisionRange(lastProcessedRevision, now)

        val committedIssues = getUnprocessedCommittedIssues(startRevision, endRevision)
        updateAffectedApps(committedIssues)

        updateTestedApps(committedIssues)

        moveToAwaitingRelease(committedIssues)

        val appNameToDeployedTag = directAppsConfProvider.getDirectAppsConf()
            .map { it.trackerAffectedApp to it.trackerDeployedTag }
            .toMap()
        moveToClosed(appNameToDeployedTag)

        val glitchedIssues = getGlitchedIssuesFromStartrek()
        moveToAwaitingRelease(glitchedIssues)

        propertiesLightSupport[LAST_PROCESSED_REVISION_PROPERTY] = endRevision.toString()
    }

    /**
     * Получить еще не обработанные (не [CLOSED_DISPLAY_STATUS] и не [AWAITING_RELEASE_DISPLAY_STATUS]) тикеты,
     * по которым были коммиты в Аркадию между [fromRevision] и [toRevision]
     *
     * @return приложение в формате [CommittedApp] -> тикеты, по которым были коммиты в этом приложении
     */
    private fun getUnprocessedCommittedIssues(
        fromRevision: SVNRevision,
        toRevision: SVNRevision,
    ): Map<CommittedApp, List<Issue>> {
        if (fromRevision.number >= toRevision.number) {
            logger.info("Skipping. Fast start")
            return emptyMap()
        }

        val arcadiaIssuesWithApp = getArcadiaCommittedIssues(fromRevision, toRevision)
        val issues = arcadiaIssuesWithApp.keys

        return issues
            .chunked(MAX_ISSUES_STARTREK_FILTER_COUNT)
            .flatMap { getUnprocessedIssuesFromStartrek(it) }
            .apply { logger.info("Found $size unprocessed issues with commits") }
            .groupBy { arcadiaIssuesWithApp.getOrDefault(it.key, CommittedApp.DIRECT) }
            .onEach { (committedApp, issues) ->
                logger.info("Found ${issues.size} issues with ${committedApp.appName} commit")
                issues.forEach { issue ->
                    logger.info("Found issue ${issue.key}: status -- ${issue.status.display}, commit -- ${committedApp.appName}, affected apps -- ${issue.affectedApps}")
                }
            }
    }

    /**
     * Получить список тикетов, по которым были коммиты в Аркадию в [DIRECT_REPOSITORY] и [FRONTEND_REPOSITORY]
     * между ревизиями [startRevision] и [endRevision]
     */
    private fun getArcadiaCommittedIssues(
        startRevision: SVNRevision,
        endRevision: SVNRevision,
    ): Map<String, CommittedApp> {
        logger.info("Get svn commits from $DIRECT_REPOSITORY between $startRevision and $endRevision revisions")
        val issues = mutableMapOf<String, CommittedApp>()
        val directSvnUrl = SVNURL.parseURIEncoded(DIRECT_REPOSITORY)
        svnClientManager.logClient.doLog(
            directSvnUrl, null, startRevision, startRevision, endRevision, false, true, 10000
        ) { log ->
            val name = Utils.getTicketByCommitMessage(log.message)
            name?.let {
                val paths: Collection<SVNLogEntryPath> = log.changedPaths.values
                when {
                    paths.any { it.path.contains("/trunk/arcadia/direct/perl") } -> {
                        issues[name] = CommittedApp.PERL
                    }
                    isFilteredPath(paths.map { it.path }) -> {
                        logger.info("Skip $name: commit to filtered directories")
                    }
                    else -> {
                        issues[name] = CommittedApp.DIRECT
                    }
                }
            }
        }
        val frontendSvnUrl = SVNURL.parseURIEncoded(FRONTEND_REPOSITORY)
        svnClientManager.logClient.doLog(
            frontendSvnUrl, null, startRevision, startRevision, endRevision, false, true, 10000
        ) { log ->
            val name = Utils.getTicketByCommitMessage(log.message)
            name?.let {
                val paths: Collection<SVNLogEntryPath> = log.changedPaths.values
                when {
                    paths.any { it.path.contains("/trunk/arcadia/adv/frontend/services/dna") } -> {
                        issues[name] = CommittedApp.DNA
                    }
                    paths.any { it.path.contains("/trunk/arcadia/adv/frontend/services/uac") } -> {
                        issues[name] = CommittedApp.UAC
                    }
                    isFilteredPath(paths.map { it.path }) -> {
                        logger.info("Skip $name: commit to filtered directories")
                    }
                    else -> {
                        issues[name] = CommittedApp.DIRECT
                    }
                }
            }
        }
        return issues
    }

    /**
     * Проверяет, что все переданные пути лежат в директориях, коммиты в которые мы не хотим обрабатывать
     */
    private fun isFilteredPath(paths: Collection<String>) = paths.all { path ->
        FILTERED_PATHS.any { filtered_path ->
            path.contains(filtered_path)
        }
    }

    /**
     * Получить информацию о тикетах [issues] из Стартрека, отфильтровав уже обработанные
     */
    private fun getUnprocessedIssuesFromStartrek(issues: List<String>): List<Issue> {
        val issuesFilter = issues.joinToString(" OR ", "(", ")") { "Key: $it" }
        val filter = """
            Queue: ${StartrekQueue.DIRECT}
            AND ($issuesFilter)
            AND Status: !${StartrekDisplayStatus.CLOSED} AND Status: !"${StartrekDisplayStatus.AWAITING_RELEASE}"
            AND "Tags": !"$WORKFLOW_GLITCH_TAG"
            """.trimIndent()
        return startrek.session.issues().find(filter).toList()
    }

    /**
     * Обновить [AFFECTED_APPS] у тикетов, если не установлено [NONE_AFFECTED_APP]
     *
     * В противном случае в тикете напишется комментарий с привызом исполнителя тикета о нарушении оформления тикета
     */
    private fun updateAffectedApps(issues: Map<CommittedApp, List<Issue>>) {
        issues[CommittedApp.DNA]?.forEach { updateAffectedApps(it, CommittedApp.DNA.appName) }
        issues[CommittedApp.UAC]?.forEach { updateAffectedApps(it, CommittedApp.UAC.appName) }
        issues[CommittedApp.PERL]?.forEach { updateAffectedApps(it, CommittedApp.PERL.appName) }
    }

    private fun updateAffectedApps(issue: Issue, app: String) {
        if (issue.affectedApps.contains(NONE_AFFECTED_APP)) {
            val comment = if (VALID_STATUSES_AFTER_COMMIT.contains(issue.status.key)) {
                """
                Привет!
                У тебя есть коммит в приложение $app по этому тикету, но в Affected apps установлено none.
                Пожалуйста, заполни Affected apps правильно.
                $DOCUMENTATION_WORKFLOW_TEXT
                """.trimIndent()
            } else {
                """
                Привет!
                У тебя есть коммит в приложение $app по этому тикету, но ты забыл(а) поменять статус тикета и Affected apps.
                Пожалуйста, заполни Affected apps правильно и переведи тикет в статус ${StartrekDisplayStatus.BETA_TESTED} или ${StartrekDisplayStatus.CLOSED}, если ручное тестирование на ТС не требуется.
                $DOCUMENTATION_WORKFLOW_TEXT
                """.trimIndent()
            }
            notifyWrongIssueState(issue, comment)
        } else {
            startrek.appendAffectedApp(issue, app)
            logger.info("Add $app to ${StartrekField.AFFECTED_APPS} for ${issue.key}")
        }
    }

    private fun updateTestedApps(issues: Map<CommittedApp, List<Issue>>) {
        issues[CommittedApp.UAC]?.forEach { updateTestedApps(it, CommittedApp.UAC.appName) }
    }

    private fun updateTestedApps(issue: Issue, app: String) {
        startrek.appendTestedApp(issue, app)
        logger.info("Add $app to ${StartrekField.TESTED_APPS} for ${issue.key}")
    }

    /**
     * Перевести тикеты в статус [AWAITING_RELEASE_DISPLAY_STATUS],
     * если тикет находится в статусе [VALID_STATUSES_FOR_AWAITING_RELEASE_MOVE]
     *
     * Тикеты с [NONE_AFFECTED_APP] в [AFFECTED_APPS] игнорируются
     *
     * В противном случае в тикете напишется комментарий с привызом исполнителя тикета о нарушении оформления тикета
     */
    private fun moveToAwaitingRelease(committedIssues: Map<CommittedApp, List<Issue>>) {
        for ((app, issues) in committedIssues) {
            for (issue in issues) {
                moveToAwaitingRelease(issue) { handleWrongStateForAwaitingReleaseMoving(issue, app.appName) }
            }
        }
    }

    /**
     * Перевести тикеты в статус [AWAITING_RELEASE_DISPLAY_STATUS],
     * если тикет находится в статусе [VALID_STATUSES_FOR_AWAITING_RELEASE_MOVE]
     *
     * Тикеты с [NONE_AFFECTED_APP] в [AFFECTED_APPS] игнорируются
     *
     * В противном случае ничего не делать
     */
    private fun moveToAwaitingRelease(issues: List<Issue>) {
        logger.info("Found ${issues.size} issues for repeated ${StartrekDisplayStatus.AWAITING_RELEASE} moving")
        for (issue in issues) {
            moveToAwaitingRelease(issue) {
                logger.info("Found wrong state for ${issue.key}: status -- ${issue.status.key}, affected apps -- ${issue.affectedApps}")
            }
        }
    }

    private fun moveToAwaitingRelease(issue: Issue, wrongStateHandler: () -> Unit) {
        if (issue.affectedApps.contains(NONE_AFFECTED_APP)) {
            logger.info("Don't need to move ${StartrekDisplayStatus.AWAITING_RELEASE}: ${issue.key} has affectedApps = $NONE_AFFECTED_APP")
            return
        }
        if (VALID_STATUSES_FOR_AWAITING_RELEASE_MOVE.contains(issue.status.key)) {
            startrek.updateStatus(issue, StartrekStatusKey.AWAITING_RELEASE)
            val comment = """
                Тикет переведен в ${StartrekDisplayStatus.AWAITING_RELEASE} и будет автоматически закрыт после выкладки во всех затронутых приложениях.
                Пожалуйста, не переводите в ${StartrekDisplayStatus.CLOSED} вручную.
                """.trimIndent()
            startrek.addComment(issue, comment, notify = false)
            logger.info("Set status = ${StartrekDisplayStatus.AWAITING_RELEASE} for ${issue.key}")
        } else {
            wrongStateHandler.invoke()
        }
    }

    private fun handleWrongStateForAwaitingReleaseMoving(issue: Issue, app: String) {
        val commitPlace = if (app == CommittedApp.DIRECT.appName) {
            "каталог /$app"
        } else {
            "приложение $app"
        }
        val comment = """
            Привет!
            У тебя есть коммит в $commitPlace по этому тикету, но ты забыл(а) поменять статус тикета.
            Пожалуйста, переведи тикет в статус ${StartrekDisplayStatus.BETA_TESTED} или ${StartrekDisplayStatus.CLOSED}, если ручное тестирование на ТС не требуется.
            $DOCUMENTATION_WORKFLOW_TEXT
            """.trimIndent()
        notifyWrongIssueState(issue, comment)
    }

    /**
     * Уведомить о нарушении оформления тикета [issue]:
     *
     * - написать комментарий [comment] с призывом исполнителя (если его нет, то автору)
     * - выставить тэг [WORKFLOW_GLITCH_TAG]
     */
    private fun notifyWrongIssueState(issue: Issue, comment: String) {
        startrek.addComment(issue, comment, summonee = issue.commentSummonee)
        startrek.addTag(issue, WORKFLOW_GLITCH_TAG)
        logger.info("Found wrong state for ${issue.key}: status -- ${issue.status.key}, affected apps -- ${issue.affectedApps}")
    }

    /**
     * Закрыть тикет, если для каждого приложения из [AFFECTED_APPS] есть тэг выложенности этого приложения
     */
    private fun moveToClosed(appNameToDeployedTag: Map<String?, String?>) {
        val filter = """
            Queue: ${StartrekQueue.DIRECT}
            AND Status: "${StartrekDisplayStatus.AWAITING_RELEASE}"
            """.trimIndent()
        val issues = startrek.session.issues().find(filter).toList()
        logger.info("Found ${issues.size} issues with ${StartrekDisplayStatus.AWAITING_RELEASE} status")
        for (issue in issues) {
            when {
                issue.affectedApps.isEmpty() -> {
                    logger.info("Don't need to move ${StartrekDisplayStatus.CLOSED}: ${issue.key} has empty affected apps")
                }
                issue.hasAllDeployedTags(appNameToDeployedTag) -> {
                    val comment = """
                        Закрываем тикет, так как он отмечен выложенным во всех затронутых приложениях.
                        Заафекченные приложения: ${issue.affectedApps}; выехало в продакшен приложений ${issue.affectedApps}.
                        """.trimIndent()
                    startrek.addComment(issue, comment, notify = false)
                    startrek.closeIssue(issue)
                    logger.info("Set status = ${StartrekDisplayStatus.CLOSED} for ${issue.key}")
                }
                else -> {
                    logger.info("Not enough deployed tags for ${issue.key}: tags -- ${issue.tags}, affected apps -- ${issue.affectedApps}")
                }
            }
        }
    }

    /**
     * Получить информацию о тикетах из Стартрека, выбрав такие, которые имеют тэг [WORKFLOW_GLITCH_TAG],
     * но готовы к повторной попытке перевода в [AWAITING_RELEASE_DISPLAY_STATUS], то есть:
     * - были поправлены исполнителем и имеют статус [VALID_STATUSES_FOR_AWAITING_RELEASE_MOVE]
     * - не содержат [NONE_AFFECTED_APP] в [AFFECTED_APPS]
     */
    private fun getGlitchedIssuesFromStartrek(): List<Issue> {
        val statusFilter = VALID_STATUSES_FOR_AWAITING_RELEASE_MOVE.joinToString(" OR ", "(", ")") { "Status: $it" }
        val filter = """
            Queue: ${StartrekQueue.DIRECT}
            AND ($statusFilter)
            AND "Tags": "$WORKFLOW_GLITCH_TAG"
            AND "Affected Apps": !"$NONE_AFFECTED_APP"
            """.trimIndent()
        return startrek.session.issues().find(filter).toList()
    }

    /**
     * Как начало отрезка выбираем первый коммит после [prevRevision].
     * Концом отрезка выбирается коммит, сделанный не позже [toTime]
     */
    private fun getRevisionRange(prevRevision: Long, toTime: String): Pair<SVNRevision, SVNRevision> {
        val svnUrl = SVNURL.parseURIEncoded(ARCADIA_REPOSITORY)
        val svnRepository = svnClientManager.repositoryPool.createRepository(svnUrl, true)
        val sdf = SimpleDateFormat(DATE_TIME_PATTERN)

        val fromRevision = prevRevision + 1
        val toRevision = svnRepository.getDatedRevision(sdf.parse(toTime))

        val startRevision = SVNRevision.create(prevRevision)
        val endRevision = SVNRevision.create(toRevision)
        logger.info("Selected revision range: (prev = r$prevRevision, toTime = $toTime) -> [r$fromRevision;r$toRevision]")
        return Pair(startRevision, endRevision)
    }
}

private enum class CommittedApp(val appName: String) {
    PERL("perl"),
    DNA("dna"),
    DIRECT("direct"),
    UAC("uac");
}
