package ru.yandex.direct.chassis.entity.startrek

import org.slf4j.LoggerFactory
import ru.yandex.direct.chassis.util.StartrekTools
import ru.yandex.direct.chassis.util.mysql.TsShardsProvider
import ru.yandex.direct.chassis.util.mysql.TsShardsProviderFactory
import ru.yandex.direct.chassis.util.startrek.StartrekComponent
import ru.yandex.direct.chassis.util.startrek.StartrekQueue
import ru.yandex.direct.chassis.util.startrek.StartrekTag
import ru.yandex.direct.scheduler.Hourglass
import ru.yandex.direct.scheduler.support.DirectJob
import ru.yandex.startrek.client.Session
import ru.yandex.startrek.client.model.Issue
import ru.yandex.startrek.client.model.IssueUpdate
import javax.annotation.ParametersAreNonnullByDefault

/**
 * Джоба, которая генерирует отчет для релиза binlogbroker
 * Срабатывает при выставленном на релизе теге
 */

@Hourglass(periodInSeconds = 120)
@ParametersAreNonnullByDefault
class BinlogbrokerTestReport(
    private val trackerClient: Session,
    tsShardsProviderFactory: TsShardsProviderFactory
) : DirectJob() {

    private val self = "https://a.yandex-team.ru/arc/trunk/arcadia/direct/apps/" +
        "chassis/src/main/java/ru/yandex/direct/chassis/entity/startrek/BinlogbrokerTestReport.java"
    private val logger = LoggerFactory.getLogger(BinlogbrokerTestReport::class.java)
    private val shardsProvider: TsShardsProvider = tsShardsProviderFactory.readWrite()

    override fun execute() {
        val query = "Queue: " + StartrekQueue.DIRECT +
            " AND Type: Release" +
            " AND Components: " + StartrekComponent.BINLOGBROKER +
            " AND Components: " + StartrekComponent.RELEASES_JAVA_DIRECT +
            " AND Status: Testing, \"RM Acceptance\"" +
            " AND Tags: " + StartrekTag.BINLOGBROKER_HEATH_CHECK
        val tickets = trackerClient.issues().find(query).toList()
        if (tickets.isEmpty()) {
            logger.info("No tickets found")
            return
        }
        logger.info("Found tickets: ${tickets.map { it.id }}")
        tickets.forEach { handleTicket(it) }
    }

    private fun handleTicket(releaseTicket: Issue) {
        logger.info("Start handle ticket ${releaseTicket.id}")
        val charts = EssCharts(shardsProvider)
        val report = """
            Графики с дашборда: $ESS_DASHBOARD_URL

            **(Посмотреть uptime бинлогброкера - с момента выкладки должна быть возврастающая прямая линия, без провалов)**
            <{Uptime бинлогброкера:
            ${charts.binlogbrokerUptimeUrl}
            }>


            **(Посмотреть binlogbroker delay - с момента выкладки отставание не должно увеличиться)**
            <{График delay'я бинлогброкера не изменился:
            ${charts.binlogbrokerDelayUrl}
            }>


            **(Посмотреть binlogbroker written messages - сильных изменений с момента выкладки быть не должно, лучше сравнивать с недельной динамикой)**
            <{График количества обработанных строк бинлогброкера после выкладки сильно не изменился:
            ${charts.binlogbrokerWrittenMessagesUrl}
            }>


            **(Посмотреть графики Delay'ев роутера и процессоров - должны быть почти нулевые графики и повторять по форму delay бинлогброкера)**
            <{Delay'и роутера и процессоров:
            ${charts.routerDelayUrl}
            ${charts.campAggregatedLastChangeDelay}
            ${charts.recomTracerDelay}
            }>


            ${charts.checkCampAggregatedLastChange()}
        """.trimIndent()

        logger.info("remove '${StartrekTag.BINLOGBROKER_HEATH_CHECK}' from tags for ticket $releaseTicket.id")
        val updateBuilder = IssueUpdate.builder().removeTags(StartrekTag.BINLOGBROKER_HEATH_CHECK)
        updateBuilder.comment(StartrekTools.signedMessageFactory(this, self, report).build())
        releaseTicket.update(updateBuilder.build())
    }
}
