package ru.yandex.direct.chassis.entity.startrek;

import java.util.Collection;
import java.util.List;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.EntryStream;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;

import ru.yandex.direct.scheduler.Hourglass;
import ru.yandex.direct.scheduler.support.DirectJob;
import ru.yandex.startrek.client.Session;
import ru.yandex.startrek.client.error.StartrekClientException;
import ru.yandex.startrek.client.model.ChecklistItem;
import ru.yandex.startrek.client.model.CommentCreate;
import ru.yandex.startrek.client.model.Issue;
import ru.yandex.startrek.client.model.IssueUpdate;

import static ru.yandex.direct.chassis.configuration.StartrekConfigurationKt.MAINTENANCE_HELPERS_TRACKER_BEAN;
import static ru.yandex.direct.chassis.util.StartrekTools.signedMessageFactory;

/**
 * Добавляет к тикету чек-лист для самопроверки при выполнении задач, затрагивающших транспорт в БК
 */
@Hourglass(periodInSeconds = 120)
@ParametersAreNonnullByDefault
public class BsTransportChecklistAdder extends DirectJob {
    private static final Logger logger = LoggerFactory.getLogger(BsTransportChecklistAdder.class);

    private static final String WIKI = "https://wiki.yandex-team.ru";
    private static final String TAG = "add_bstransport_checklist";

    private static final String SELF = "https://a.yandex-team.ru/arc/trunk/arcadia/direct/apps" +
            "/chassis/src/main/java/ru/yandex/direct/chassis/entity/startrek/BsTransportChecklistAdder.java";
    private static final String TOOL_MOVE = "https://direct.yandex.ru" +
            "/internal_tools/#move_campaigns_to_bs_export_specials";
    private static final String LOGVIEWER = "https://direct.yandex.ru/logviewer/#bsexport_data";
    private static final String LIMTEST_HOWTO = WIKI + "/direct/development/howto/procedure-limtest/";
    private static final String ABOUT_AUTO_SUPBS_DIFF = WIKI +
            "/direct/development/howto/show-auto-supbs-diff-in-issues/";
    private static final String DUTY = "https://abc.yandex-team.ru/services/direct/duty/?role=3298&interval=P1M";
    private static final String TEST_PACK = "https://aqua.yandex-team.ru/#/pack/5509b4f7e4b07e43030816b9";
    private static final String TASK_PREFIX = "[тикет]";
    private static final String SPEC_PREFIX = "[формат данных]";
    private static final String LIMTEST_PREFIX = "[тестирование на limtest]";

    private static final List<String> DEVELOPER_CHECKLIST = List.of(
            "зафиксировать и согласовать " + SPEC_PREFIX,
            "написать код",
            "попробовать на бете отправку данных в фейковую БК, приложить лог",
            "прогнать регрессию на бете (уменьшив параллельность до 70) " + TEST_PACK,
            "исправить функциональные автотесты, написать новые или запланировать это",
            "подумать о расширении покрытия b2b тестов",
            "соответствующим образом оформить " + TASK_PREFIX,
            "оценить (и записать в тикет!) влияние изменений на автобюджетные заказы," +
                    " подробнее в https://st.yandex-team.ru/DIRECTKNOL-45",
            "пройти " + LIMTEST_PREFIX,
            "перед мержем в транк, если задача у существующих кампаний меняет значения полей или добавляет новые" +
                    ", то это влияет на b2b тесты и нужно предупредить дежурных " + DUTY,
            "смержить изменения в тестах, проверить что новые включены в пак",
            "смержить задачу в транк"
    );

    private static final List<String> TASK_CHECKLIST = List.of(
            "отвечает на вопрос \"для чего эти изменения? какую задачу решаем?\"",
            "к нему приликована соответствующая задача БК",
            "в случае добавления новых полей в bssoap ДОЛЖНА быть прилинкована задача БК" +
                    " по переводу их обработки на контент-систему",
            "в описании максимально подробно зафиксирован формат отправляемых данных, их пример",
            "в нем зафиксировано, куда в БК раскладываются присылаемые данные (мастербаза, таблицы в YTе)",
            "в описании указаны сценарии или типы объектов, для которых применимы изменения",
            "в нем получен комментарий от БК об их готовности в продакшене принимать" +
                    " (корректно обрабатывать или игнорировать) наши изменения",
            "при пропуске тестирования на limtest дан комментарий, объясняющий этот шаг"
    );

    private static final List<String> SPECIFICATION_CHECKLIST = List.of(
            "название полей записано одинаково в Директе и БК (с учетом регистра)",
            "совпадает тип значения в Директе и БК: скаляр, массив или словарь (+ вложенные структуры)",
            "скалярные значения ожидаются как строки или как числа?",
            "отметить, нужно ли явное приведение типа для скаляров? (например 0 -> \"0\")",
            "отметить, возможны ли дробные значения и нужно ли округление? (для числовых значений)",
            "согласовано поведение для опциональных полей при отсутствии значения" +
                    " (варианты: отсутствие ключа, NULL, специальное значение)",
            "для всех (не только новых) полей зафиксировано, что в БК является их источником: фид Директа или bssoap?",
            "значения полей, принимаемых БК только из фида, обернуты в коде экспорта в '_nosoap()'" +
                    " или вложены в струтктуру, обернутую в '_nosoap()'"
    );

    private static final List<String> LIMTEST_CHECKLIST = List.of(
            "найти подходящую кампанию или создать (https://st.yandex-team.ru/DIRECTKNOL-43)",
            "убедиться, что кампания и все её объекты приняты на модерации",
            "убедиться, что на кампании или общем счете есть деньги",
            "если денег нет и клиент тестовый — заказать сертификатные средства" +
                    " (https://st.yandex-team.ru/DIRECTKNOL-42)",
            "выбрать с каким стендом БК тестируем: прод или препрод bssoap (https://st.yandex-team.ru/DIRECTKNOL-44)" +
                    ", записать в комментариях выбор",
            "переотправить кампанию в БК без внесения изменений или продакшн-кодом",
            "проверить по логам успешность первой отправки (неотредактированной) кампании " + LOGVIEWER,
            "убедиться, что кампания не отправляется бесконечно (много логов) - логи должны заканчиваться",
            "проверить, что в логах ответов (response) нет Error, UnDone",
            "убедиться, что отправились всё что мы хотели: кампания, группа, баннер",
            "добавить OrderID кампании в описание тикета в формате БК: ХХХ",
            "поставить на тикет тег auto_supbs, дождаться комментария робота",
            "выложить код с изменениями на стенд по инструкции " + LIMTEST_HOWTO,
            "переместить кампанию в соответствующую (номеру лимтеста) очередь экспорта DEV1 или DEV2 " + TOOL_MOVE,
            "внести проверяемые изменения в кампанию либо просто переотправить её в БК",
            "проверить по логам успешность второй отправки " + LOGVIEWER,
            "убедиться, что времена событий в логе больше, чем при первой отправке",
            "убедиться, что кампания не переотправляется бесконечно - логи должны заканчиваться",
            "проверить, что в логах ответа на второй запрос нет Error, UnDone",
            "убедиться, что отправились всё что мы хотели: кампания, группа, баннер",
            "убедиться, что в логе экспорта есть ожидаемые нами изменения",
            "снова поставить на тикет тег auto_supbs, дождаться комментария робота",
            "поставить на тикет тег show_auto_supbs_diff, дождаться комментария робота с результатом сравнения "
                    + ABOUT_AUTO_SUPBS_DIFF,
            "проанализировать изменения в данных БК",
            "убедиться что есть ожидаемые изменения и нет неожиданных",
            "добавить в тикет выборку данных из контет-системы (если требуется)",
            "убедиться, что кампания доехала до выдачи / показывается / есть статистика (при необходимости)",
            "остановить кампанию (для кампаний тестовых клиентов или созданых под проверку)",
            "вернуть кампанию обратно в стандартную очередь " + TOOL_MOVE
    );

    private static final List<ChecklistItem> CHECKLIST = EntryStream.of(
            "", DEVELOPER_CHECKLIST,
            TASK_PREFIX, TASK_CHECKLIST,
            SPEC_PREFIX, SPECIFICATION_CHECKLIST,
            LIMTEST_PREFIX, LIMTEST_CHECKLIST
    )
            .flatMapValues(Collection::stream)
            .mapKeyValue(BsTransportChecklistAdder::concatWithPrefix)
            .map(item -> new ChecklistItem.Builder().text(item).build())
            .toList();
    private final Session client;

    @Autowired
    public BsTransportChecklistAdder(@Qualifier(MAINTENANCE_HELPERS_TRACKER_BEAN) Session client) {
        this.client = client;
    }

    private static String concatWithPrefix(String prefix, String text) {
        return prefix + " " + text;
    }

    @Override
    public void execute() {
        logger.debug("fetch unprocessed tickets from tracker");
        client.issues()
                .find("Queue: DIRECT AND Tags: " + TAG)
                .forEachRemaining(this::processIssue);
        logger.debug("all tickets processed");
    }

    private void processIssue(Issue issue) {
        try {
            processIssueUnsafe(issue);
        } catch (StartrekClientException e) {
            logger.error("Failed to update issue", e);
        }
    }

    private void processIssueUnsafe(Issue issue) {
        logger.info("Processing issue {}", issue.getKey());

        IssueUpdate.Builder updateBuilder = IssueUpdate.builder().removeTags(TAG);
        CommentCreate comment;
        if (issue.getChecklist().isEmpty()) {
            logger.debug("Add checklist to issue");
            issue.addChecklistItems(BsTransportChecklistAdder.CHECKLIST);
            comment = getSignedMessage("В тикет добавлен чеклист по транспорту в БК." +
                    "\nОн составлен из важных для разработки и тестирования пунктов." +
                    " Используйте его для самопроверки: отмечайте галочкой пройденные пункты," +
                    " пропущенные - сопровождайте комментарием и тоже отмечайте");

        } else {
            logger.debug("Issue already has checklist, skip adding");
            comment = getSignedMessage("Чеклист по транспорту в БК не добавлен: в тикете уже есть чеклист");
        }
        updateBuilder.comment(comment);
        issue.update(updateBuilder.build());
    }

    private CommentCreate getSignedMessage(String message) {
        return signedMessageFactory(this, SELF, message).build();
    }
}
