package ru.yandex.direct.chassis.entity.startrek

import org.slf4j.LoggerFactory
import ru.yandex.direct.chassis.util.DEV_DEPLOY_STAGES
import ru.yandex.direct.chassis.util.DeployService
import ru.yandex.direct.chassis.util.DeployStage
import ru.yandex.direct.chassis.util.DirectAppsConfEntry
import ru.yandex.direct.chassis.util.DirectAppsConfProvider
import ru.yandex.direct.chassis.util.DirectReleaseUtils
import ru.yandex.direct.chassis.util.ReleaseVersion
import ru.yandex.direct.chassis.util.nonCf
import ru.yandex.direct.chassis.util.startrek.ChecklistItemUpdate
import ru.yandex.direct.chassis.util.startrek.StartrekHelper
import ru.yandex.direct.chassis.util.startrek.StartrekStatusKey
import ru.yandex.direct.env.ProductionOnly
import ru.yandex.direct.scheduler.Hourglass
import ru.yandex.direct.scheduler.support.DirectJob
import ru.yandex.startrek.client.model.ChecklistItem
import ru.yandex.startrek.client.model.Issue

/**
 * Обновлятор чеклиста релиза. Последовательно проверяет пункты чеклиста
 * и автоматически расставляет им галки когда требуется.
 *
 * Сам чеклист добавляется в релизный тикет джобой `AutocommentReleasesJob`.
 *
 * См. описание в [DIRECT-155346](https://st.yandex-team.ru/DIRECT-155346)
 */
@Hourglass(periodInSeconds = 5 * 60, needSchedule = ProductionOnly::class)
class ChecklistReleasesJob(
    private val startrek: StartrekHelper,
    private val deployService: DeployService,
    private val directAppsConfProvider: DirectAppsConfProvider,
) : DirectJob() {

    companion object {
        private const val DISABLE_AUTO_CHECKLIST_TAG = "auto_checklist_off"
        private val logger = LoggerFactory.getLogger(ChecklistReleasesJob::class.java)
    }

    private val component2app: Map<String, DirectAppsConfEntry>
        get() = directAppsConfProvider.getDirectAppsConf()
            .filter { it.trackerComponent.isNotBlank() }
            .associateBy { it.trackerComponent }

    override fun execute() {
        val query = """
            | Queue: DIRECT
            | Type: Release
            | Components: "Releases: JavaDirect"
            | Status: Testing, ReadyForTest, "RM Acceptance"
            | Tags: "$AUTOCOMMENTS_TAG"
            | Tags: !"$DISABLE_AUTO_CHECKLIST_TAG"
            | "Sort by": key desc
        """.trimMargin()

        val releases = startrek.session.issues().find(query)
            .toList().nonCf()
//            .filter { it.key == "DIRECT-155421" } // Для тестирования
            .filter { it.checklist.isNotEmpty }

        logger.info("Found ${releases.size} release tickets to check checklists")

        for (release in releases) {
            val app = release.components.nonCf()
                .firstNotNullOfOrNull { component2app[it.display] }
                ?: continue
            app.trackerAffectedApp
                ?: continue
            processRelease(release, app)
        }
    }

    private fun processRelease(release: Issue, app: DirectAppsConfEntry) {
        release.checklist.forEach { checklistItem ->
            val container = CheckContainer(release, checklistItem, app)
            val checkResult = container.check()

            if (checkResult.checked != null && checkResult.checked != checklistItem.checked) {
                logger.info(
                    "Updating checklist item ${checklistItem.text} on release ${release.key}: " +
                        "${checklistItem.checked} -> ${checkResult.checked}"
                )

                startrek.updateChecklist(release, checklistItem.id, checkResult)
            }
        }
    }

    private fun CheckContainer.check(): ChecklistItemUpdate {
        return when (checklistItem.text) {
            UNIT_TESTS_ITEM -> checkUnitTests()
            AUTO_TESTS_ITEM -> checkAutoTests()
            MANUAL_TESTING_ITEM -> checkManualTesting()
            SORT_OUT_ITEM -> checkSortOut()
            DEV_ENVIRONMENT_ITEM -> checkDevEnvironment()
            RELEASE_MANUAL_HEALTH_CHECK -> ChecklistItemUpdate()
            else -> {
                logger.warn("Unknown checklist item: ${checklistItem.text}")
                return ChecklistItemUpdate()
            }
        }
    }

    private fun CheckContainer.checkUnitTests(): ChecklistItemUpdate {
        val unitTestIssues = DirectReleaseUtils.findUnitTestIssues(release)
        val allIssuesAreClosed = unitTestIssues
            .all { it.status.key == StartrekStatusKey.CLOSED }
        return ChecklistItemUpdate.check(allIssuesAreClosed)
    }

    private fun CheckContainer.checkAutoTests(): ChecklistItemUpdate {
        val autoTestIssues = DirectReleaseUtils.findRegressionIssues(release)
        val allIssuesAreClosed = autoTestIssues
            .all { it.status.key == StartrekStatusKey.CLOSED }
        return ChecklistItemUpdate.check(allIssuesAreClosed)
    }

    private fun CheckContainer.checkManualTesting(): ChecklistItemUpdate {
        val query = toTestQuery(app.trackerAffectedApp!!, release.key)
        val count = startrek.issuesCount(query)
        return ChecklistItemUpdate.check(count == 0)
    }

    private fun CheckContainer.checkSortOut(): ChecklistItemUpdate {
        val query = toReviewQuery(app.trackerAffectedApp!!, release.key)
        val count = startrek.issuesCount(query)
        return ChecklistItemUpdate.check(count == 0)
    }

    private fun CheckContainer.checkDevEnvironment(): ChecklistItemUpdate {
        val appAvailableStages = app
            .yaDeployStages.keys
            .map { DeployStage.getByLabel(it) }
            .toSet()

        val devDeployStages = when (app.name) {
            JAVA_LOGVIEWER_APP -> emptySet()
            JAVA_INTAPI_APP, JAVA_WEB_APP, JAVA_API5_APP -> setOf(DeployStage.DEV7)
            else -> DEV_DEPLOY_STAGES
        }

        val stagesToCheck = appAvailableStages.intersect(devDeployStages)

        val releaseVersion = DirectReleaseUtils.getReleaseVersion(release)
        if (releaseVersion == null) {
            logger.error("Failed to find release version for issue ${release.key}")
            return ChecklistItemUpdate.uncheck()
        }

        for (stage in stagesToCheck) {
            val stageNameList = app.yaDeployStages[stage.label]
            val stageStatusList = stageNameList?.map { deployService.getDeployStatusSummary(app, it, stage.label) }

            if (stageStatusList != null) {
                for (stageStatus in stageStatusList) {
                    val stageVersion = ReleaseVersion.parse(stageStatus.version)

                    if (stageVersion == null
                        || stageVersion < releaseVersion
                        || (stageVersion == releaseVersion && !stageStatus.isDeployTicketClosed)
                    ) {

                        return ChecklistItemUpdate.uncheck()
                    }
                }
            }
        }

        return ChecklistItemUpdate.check()
    }

    private data class CheckContainer(
        val release: Issue,
        val checklistItem: ChecklistItem,
        val app: DirectAppsConfEntry,
    )
}
