package ru.yandex.direct.chassis.entity.startrek

import org.apache.http.client.utils.URIBuilder
import org.jooq.impl.DSL
import org.slf4j.LoggerFactory
import ru.yandex.direct.chassis.util.mysql.TsShardsProvider
import ru.yandex.direct.dbschema.ppc.Tables
import java.net.URI
import java.time.Clock
import java.time.Duration
import java.time.LocalDateTime
import java.time.format.DateTimeFormatter
import java.time.temporal.ChronoUnit

class EssCharts(
    private val shardsProvider: TsShardsProvider
) {
    private val logger = LoggerFactory.getLogger(EssCharts::class.java)

    private val now = LocalDateTime.now(Clock.systemUTC())
    private val dateFrom = DateTimeFormatter.ISO_DATE_TIME.format(now.minusDays(1))
    private val dateTo = DateTimeFormatter.ISO_DATE_TIME.format(now.plusDays(1))

    private val essRouterWithDates: URI = URIBuilder(ESS_ROUTER_COMMON_URL)
        .addParameter("b", dateFrom + "Z")
        .addParameter("e", dateTo + "Z")
        .build()

    private val binlogbrokerWithDates: URI = URIBuilder(BINLOGBROKER_COMMON_URL)
        .addParameter("b", dateFrom + "Z")
        .addParameter("e", dateTo + "Z")
        .build()

    val routerUptimeUrl = URIBuilder(essRouterWithDates)
        .addParameter("l.sensor", "jvm.runtime.uptime")
        .addParameter("l.router_chunk_id", "*")
        .build().toIframe()

    val binlogbrokerUptimeUrl = URIBuilder(binlogbrokerWithDates)
        .addParameter("l.sensor", "work_time")
        .build().toIframe()

    val binlogbrokerDelayUrl = URIBuilder(binlogbrokerWithDates)
        .addParameter("l.sensor", "logbrokerwriter_delay_seconds")
        .addParameter("stack", "false")
        .build().toIframe()

    val routerDelayUrl = URIBuilder(essRouterWithDates)
        .addParameter("l.sensor", "binlog_delay")
        .addParameter("l.shard", "*")
        .addParameter("stack", "false")
        .addParameter("l.router_chunk_id", "*")
        .build().toIframe()

    val binlogbrokerWrittenMessagesUrl = URIBuilder(binlogbrokerWithDates)
        .addParameter("l.sensor", "logbrokerwriter_written_messages")
        .addParameter("transform", "differentiate")
        .build().toIframe()

    val routerWrittenMessagesUrl = URIBuilder(essRouterWithDates)
        .addParameter("l.sensor", "handled_binlog_rows_count")
        .addParameter("l.shard", "*")
        .addParameter("l.router_chunk_id", "*")
        .addParameter("transform", "differentiate")
        .build().toIframe()

    val campAggregatedLastChangeDelay = URIBuilder(PROCESSORS_DELAY_COMMON_URL)
        .addParameter("l.ess_processor", "camp_aggregated_last_change")
        .addParameter("b", dateFrom + "Z")
        .addParameter("e", dateTo + "Z")
        .build().toIframe()

    val recomTracerDelay = URIBuilder(PROCESSORS_DELAY_COMMON_URL)
        .addParameter("l.ess_processor", "recom_tracer")
        .addParameter("b", dateFrom + "Z")
        .addParameter("e", dateTo + "Z")
        .build().toIframe()

    private fun URI.toIframe() =
        "{{iframe height='400' width='100%' src='$this' frameborder='0'}}"

    fun checkCampAggregatedLastChange(): String {
        val now = LocalDateTime.now().truncatedTo(ChronoUnit.SECONDS)
        logger.info("Execute query: update banners set lastchange = '$now' limit 1")

        shardsProvider.get().forEach { dbWrapper ->
            dbWrapper.dslContext.update(Tables.BANNERS)
                .set(Tables.BANNERS.LAST_CHANGE, now)
                .limit(1)
                .execute()
        }
        logger.info("Sleeping")
        Thread.sleep(Duration.ofSeconds(10).toMillis())
        var isGood = true
        logger.info("Fetch max last change from camp_aggregated_last_change")
        val resultOnShards = shardsProvider.get().map { dbWrapper ->
            val maxLastChangeColumn = DSL.max(Tables.CAMP_AGGREGATED_LASTCHANGE.LAST_CHANGE).`as`("max_last_change")
            val maxLastChange = dbWrapper.dslContext
                .select(maxLastChangeColumn)
                .from(Tables.CAMP_AGGREGATED_LASTCHANGE)
                .fetchOne(maxLastChangeColumn)
            dbWrapper.dbname to maxLastChange
        }.joinToString("\n") {
            val isGoodOnShard = !it.second.isBefore(now)
            val resultStr = "${it.first}: max(last_change) = ${it.second} - ${if (isGoodOnShard) ok() else fail()}"
            isGood = isGoodOnShard && isGood
            resultStr
        }

        return """
            **(Проверить работу CampAggregatedLastchangeProcessor - процессор должен быстро (в течение нескольких секунд) подхватить изменения баннеров)**
            <{CampAggregatedLastchangeProcessor - ${if (isGood) ok() else fail()}:
            **Триггер для процессора:**
            %%update banners set lastchange = $now limit 1\G%%
            **Проверка:**
            $resultOnShards
            }>
        """.trimIndent()
    }

    private fun ok() = "!!(зел)OK!!"
    private fun fail() = "!!FAIL!!"
}

private const val SOLOMON_URL = "https://solomon.yandex-team.ru/"
val ESS_DASHBOARD_URL: URI = URIBuilder(SOLOMON_URL)
    .addParameter("cluster", "app_binlogbroker")
    .addParameter("service", "java-monitoring")
    .addParameter("env", "testing")
    .addParameter("dashboard", "ess-test")
    .addParameter("project", "direct-test")
    .build()

val ESS_ROUTER_COMMON_URL: URI = URIBuilder(SOLOMON_URL)
    .addParameter("project", "direct-test")
    .addParameter("cluster", "app_ess-router")
    .addParameter("service", "java-monitoring")
    .addParameter("l.host", "CLUSTER")
    .addParameter("l.env", "testing")
    .addParameter("graph", "auto")
    .addParameter("graphOnly", "y")
    .build()

val BINLOGBROKER_COMMON_URL: URI = URIBuilder(SOLOMON_URL)
    .addParameter("project", "direct-test")
    .addParameter("cluster", "app_binlogbroker")
    .addParameter("service", "java-monitoring")
    .addParameter("l.host", "CLUSTER")
    .addParameter("l.env", "testing")
    .addParameter("graph", "auto")
    .addParameter("graphOnly", "y")
    .addParameter("l.binlogbroker_format", "protobuf")
    .build()

private val PROCESSORS_DELAY_COMMON_URL: URI =
    URIBuilder(SOLOMON_URL)
        .addParameter("project", "direct-test")
        .addParameter("cluster", "app_java-jobs")
        .addParameter("service", "java-monitoring")
        .addParameter("l.host", "CLUSTER")
        .addParameter("l.env", "testing")
        .addParameter("graph", "auto")
        .addParameter("graphOnly", "y")
        .addParameter("sensor", "binlog_delay")
        .addParameter("l.shard", "*")
        .addParameter("stack", "false")
        .build()
