package ru.yandex.direct.chassis.entity.startrek

import org.apache.http.client.utils.URIBuilder
import org.jooq.impl.DSL.max
import org.slf4j.LoggerFactory
import ru.yandex.direct.chassis.util.StartrekTools
import ru.yandex.direct.chassis.util.mysql.TsShardsProvider
import ru.yandex.direct.chassis.util.mysql.TsShardsProviderFactory
import ru.yandex.direct.chassis.util.startrek.StartrekComponent
import ru.yandex.direct.chassis.util.startrek.StartrekQueue
import ru.yandex.direct.chassis.util.startrek.StartrekTag
import ru.yandex.direct.scheduler.Hourglass
import ru.yandex.direct.scheduler.support.DirectJob
import ru.yandex.startrek.client.Session
import ru.yandex.startrek.client.model.Issue
import ru.yandex.startrek.client.model.IssueUpdate
import java.net.URI
import java.time.Clock
import java.time.Duration
import java.time.LocalDateTime
import java.time.format.DateTimeFormatter
import java.time.temporal.ChronoUnit
import javax.annotation.ParametersAreNonnullByDefault

/**
 * Джоба, которая генерирует отчет для релиза ess-router
 * Срабатывает при выставленном на релизе теге
 */
@Hourglass(periodInSeconds = 120)
@ParametersAreNonnullByDefault
class EssRouterTestReport(
    private val trackerClient: Session,
    tsShardsProviderFactory: TsShardsProviderFactory
) : DirectJob() {

    private val logger = LoggerFactory.getLogger(EssRouterTestReport::class.java)
    private val SELF = "https://a.yandex-team.ru/arc/trunk/arcadia/direct/apps/" +
        "chassis/src/main/java/ru/yandex/direct/chassis/entity/startrek/EssRouterTestReport.java"
    private val shardsProvider: TsShardsProvider = tsShardsProviderFactory.readWrite()

    override fun execute() {
        val query = "Queue: " + StartrekQueue.DIRECT +
            " AND Type: Release" +
            " AND Components: " + StartrekComponent.ESS_ROUTER +
            " AND Components: " + StartrekComponent.RELEASES_JAVA_DIRECT +
            " AND Status: Testing, \"RM Acceptance\"" +
            " AND Tags: " + StartrekTag.ESS_ROUTER_HEATH_CHECK
        val tickets = trackerClient.issues().find(query).toList()
        if (tickets.isEmpty()) {
            logger.info("No tickets found")
            return
        }
        logger.info("Found tickets: ${tickets.map { it.id }}")
        tickets.forEach { handleTicket(it) }
    }

    private fun handleTicket(releaseTicket: Issue) {
        logger.info("Start handle ticket ${releaseTicket.id}")
        val charts = EssCharts(shardsProvider)

        val report = """
            Графики с дашборда: $ESS_DASHBOARD_URL

            **(Посмотреть uptime роутера - должна быть возврастающая прямая линия, без провалов)**
            <{Uptime роутера:
            ${charts.routerUptimeUrl}
            }>


            **(Посмотреть binlogbroker delay и router delay - графики должны быть подобны по форме)**
            <{График delay'я роутера повторяет по форме delay бинлогброкера:
            ${charts.binlogbrokerDelayUrl}
            ${charts.routerDelayUrl}
            }>


            **(Посмотреть binlogbroker written messages и router processed rows - графики должны быть подобны по форме)**
            <{График количества обработанных сообщений роутера повторяет по форме количество обработанных строк бинлогброкера:
            ${charts.binlogbrokerWrittenMessagesUrl}
            ${charts.routerWrittenMessagesUrl}
            }>


            **(Посмотреть графики Delay'ев процессоров - должны быть почти нулевые графики)**
            <{Delay'и процессоров:
            ${charts.campAggregatedLastChangeDelay}
            ${charts.recomTracerDelay}
            }>


            ${charts.checkCampAggregatedLastChange()}
        """.trimIndent()

        logger.info("remove '${StartrekTag.ESS_ROUTER_HEATH_CHECK}' from tags for ticket $releaseTicket.id")
        val updateBuilder = IssueUpdate.builder().removeTags(StartrekTag.ESS_ROUTER_HEATH_CHECK)
        updateBuilder.comment(StartrekTools.signedMessageFactory(this, SELF, report).build())
        releaseTicket.update(updateBuilder.build())
    }
}

