package ru.yandex.direct.chassis.entity.startrek

import org.slf4j.LoggerFactory
import ru.yandex.direct.chassis.util.DirectAppsConfProvider
import ru.yandex.direct.chassis.util.calendar.HolidayService
import ru.yandex.direct.chassis.util.calendar.HolidayType
import ru.yandex.direct.chassis.util.startrek.StartrekHelper
import ru.yandex.direct.chassis.util.startrek.StartrekQueue
import ru.yandex.direct.chassis.util.startrek.StartrekStatusKey
import ru.yandex.direct.chassis.util.startrek.StartrekTag
import ru.yandex.direct.env.ProductionOnly
import ru.yandex.direct.scheduler.Hourglass
import ru.yandex.direct.scheduler.support.DirectJob
import ru.yandex.startrek.client.model.Issue
import ru.yandex.startrek.client.model.UserRef
import java.time.LocalDate

@Hourglass(cronExpression = "0 */5 10-19 * * ?", needSchedule = ProductionOnly::class)
class IssuesReminderJob(
    private val startrek: StartrekHelper,
    private val holidayService: HolidayService,
    private val directAppsConfProvider: DirectAppsConfProvider,
) : DirectJob() {
    private val logger = LoggerFactory.getLogger(IssuesReminderJob::class.java)

    override fun execute() {
        val holiday = holidayService.getHoliday(LocalDate.now())
        if (holiday.type != HolidayType.WEEKDAY) {
            logger.info("Not a working day, skipping: $holiday")
            return
        }

        directAppsConfProvider.getDirectAppsConf().forEach { app ->
            val query = """
                Queue: ${StartrekQueue.DIRECT}, ${StartrekQueue.TESTIRT}
                Status: !${StartrekStatusKey.CLOSED}
                Tags: ${StartrekTag.RELEASE_TESTING} Tags: ${StartrekTag.releaseTesting(app.name)}
                Tags: !${StartrekTag.NOTIFICATION_DONE} Tags: !${StartrekTag.notificationDone(app.name)}
                Tags: !${StartrekTag.DONT_REMIND}
                ${app.trackerAffectedApp?.let { "\"Tested apps\": !$it" }.orEmpty()}
                (Type: !bug OR (Type: bug AND Created: < today() - 1h))
            """.trimIndent()
            val issues = startrek.findIssues(query)

            val issuesToCall: Map<Issue, UserRef> = issues
                .associateWith { issue -> issue.qaEngineer ?: issue.assignee.orNull ?: issue.createdBy }
                .filter { (_, callee) -> callee.login != "ppc" }

            issuesToCall.forEach { (issue, callee) ->
                startrek.addComment(
                    issue,
                    """
                        Привет, тебя ждет задача в релизе приложения ${app.name}!
                        Напоминаем, что тикеты в релизах имеют приоритет над другими.
                        Ожидается, что ты сделаешь все что надо в течение 3 часов от начала релиза (время между 19:00 и 9:00 не учитывается).
                        Подробнее: https://docs.yandex-team.ru/direct-dev/releases/release-ticket-call
                    """.trimIndent(),
                    summonee = callee,
                    ifNoTag = StartrekTag.notificationDone(app.name),
                )
            }
        }
    }
}
