package ru.yandex.direct.chassis.entity.startrek;

import javax.annotation.ParametersAreNonnullByDefault;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;

import ru.yandex.direct.chassis.util.startrek.StartrekComponent;
import ru.yandex.direct.chassis.util.startrek.StartrekQueue;
import ru.yandex.direct.chassis.util.startrek.StartrekStatusKey;
import ru.yandex.direct.chassis.util.startrek.StartrekTag;
import ru.yandex.direct.scheduler.Hourglass;
import ru.yandex.direct.scheduler.support.DirectJob;
import ru.yandex.startrek.client.Session;
import ru.yandex.startrek.client.model.CommentCreate;
import ru.yandex.startrek.client.model.Issue;
import ru.yandex.startrek.client.model.IssueUpdate;

import static ru.yandex.direct.chassis.configuration.StartrekConfigurationKt.MAINTENANCE_HELPERS_TRACKER_BEAN;
import static ru.yandex.direct.chassis.util.StartrekTools.signedMessageFactory;

@Hourglass(periodInSeconds = 300)
@ParametersAreNonnullByDefault
public class JobsReleaseToTesting extends DirectJob {
    private static final Logger logger = LoggerFactory.getLogger(JobsReleaseToTesting.class);
    private static final String SELF = "https://a.yandex-team.ru/arc/trunk/arcadia/direct/apps/" +
            "chassis/src/main/java/ru/yandex/direct/chassis/entity/startrek/JobsReleaseToTesting.java";
    private static final String QUEUE = StartrekQueue.DIRECT;
    private static final String STATUS_TESTING = StartrekStatusKey.TESTING;

    private final Session client;

    @Autowired
    public JobsReleaseToTesting(@Qualifier(MAINTENANCE_HELPERS_TRACKER_BEAN) Session client) {
        this.client = client;
    }

    @Override
    public void execute() {
        logger.debug("'jobs release to testing' task started");
        logger.debug("finding release ticket");
        String query = "Queue: " + QUEUE +
                " AND Type: Release" +
                " AND Components: " + StartrekComponent.JAVA_JOBS +
                " AND Components: " + StartrekComponent.RELEASES_JAVA_DIRECT +
                " AND Status: \"Ready For Test\"";
        client.issues().find(query).forEachRemaining(this::report);

        logger.debug("'jobs release to testing' task completed");
    }


    private void report(Issue ticket) {
        logger.info("release ticket founded: {}", ticket);
        IssueUpdate.Builder updateBuilder = IssueUpdate.builder();
        CommentCreate comment;
        StringBuilder sb = new StringBuilder("Тикет автоматически переведен в статус \"Тестируется\"\n\n");
        sb.append("**Краткая инструкция по тестированию релиза jobs:**\n");
        sb.append("Протестируй входящие в релиз задачи.\n");
        sb.append("Подожди после обновления ТС пару-тройку часов, поставь тег %%" + StartrekTag.CHECK_JOBS_REGRESSION +
                "%%. В течение нескольких минут робот напишет комментарий \"Проверено на ТС по чеклисту ...\" и, " +
                "возможно, прилинкует к релизу тикеты о проблемах с джобами. " +
                "((https://wiki.yandex-team.ru/direct/development/howto/releases-jobs-java/#razbor " +
                "Проанализируй)) комментарий — в нём курсивом даны подсказки \"как должно быть\", " +
                "разбери созданные роботом тикеты (если есть).\n");
        sb.append("(опционально) Напиши комментарий о тестировании новых задач или их отсутствии.\n\n");
        sb.append("Подробнее на ((https://wiki.yandex-team.ru/direct/development/howto/releases-jobs-java/ вики)).\n");
        comment = signedMessageFactory(this, SELF, sb.toString()).build();
        client.transitions().execute(ticket.getKey(), STATUS_TESTING);
        logger.info("Transitioned to {}", STATUS_TESTING);
        updateBuilder.comment(comment);
        ticket.update(updateBuilder.build());
        logger.info("Posted release instruction");
    }
}
