package ru.yandex.direct.chassis.entity.startrek;

import java.net.URISyntaxException;
import java.text.SimpleDateFormat;
import java.time.Clock;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.Date;
import java.util.List;
import java.util.function.Function;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.node.ObjectNode;
import org.apache.commons.lang3.tuple.Pair;
import org.apache.http.client.utils.URIBuilder;
import org.jooq.DSLContext;
import org.jooq.impl.DSL;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Qualifier;

import ru.yandex.bolts.collection.IteratorF;
import ru.yandex.direct.chassis.util.YpClient;
import ru.yandex.direct.chassis.util.mysql.TsShardsProvider;
import ru.yandex.direct.chassis.util.mysql.TsShardsProviderFactory;
import ru.yandex.direct.chassis.util.startrek.StartrekComponent;
import ru.yandex.direct.chassis.util.startrek.StartrekQueue;
import ru.yandex.direct.chassis.util.startrek.StartrekTag;
import ru.yandex.direct.dbschema.ppc.Tables;
import ru.yandex.direct.juggler.JugglerChecksStateFilter;
import ru.yandex.direct.juggler.JugglerChecksStateItem;
import ru.yandex.direct.juggler.JugglerClient;
import ru.yandex.direct.liveresource.LiveResource;
import ru.yandex.direct.liveresource.LiveResourceEvent;
import ru.yandex.direct.liveresource.LiveResourceListener;
import ru.yandex.direct.scheduler.Hourglass;
import ru.yandex.direct.scheduler.support.DirectJob;
import ru.yandex.startrek.client.Session;
import ru.yandex.startrek.client.model.Issue;
import ru.yandex.startrek.client.model.IssueCreate;
import ru.yandex.startrek.client.model.IssueUpdate;

import static ru.yandex.direct.chassis.configuration.StartrekConfigurationKt.MAINTENANCE_HELPERS_TRACKER_BEAN;
import static ru.yandex.direct.chassis.util.StartrekTools.buildTable;
import static ru.yandex.direct.chassis.util.StartrekTools.signedMessageFactory;
import static ru.yandex.direct.utils.JsonUtils.fromJson;
import static ru.yandex.direct.utils.JsonUtils.toJson;

@Hourglass(periodInSeconds = 120)
@ParametersAreNonnullByDefault
public class JobsTestReport extends DirectJob {
    private static final Logger logger = LoggerFactory.getLogger(JobsTestReport.class);
    private static final String SELF = "https://a.yandex-team.ru/arc/trunk/arcadia/direct/apps/" +
            "chassis/src/main/java/ru/yandex/direct/chassis/entity/startrek/JobsTestReport.java";
    private static final String JOBS_DEPLOY_STAGE_NAME = "direct-java-jobs-test";

    private static final String LINE_SEPARATOR = "----------------\n";
    private static final String JUGGLER_HOST_DIRECT_CHECKS_TEST = "direct.checks.test";
    private static final String SOLOMON_URL = "https://solomon.yandex-team.ru/";

    private static final String QUEUE = StartrekQueue.DIRECT;
    private static final Comparator<JugglerChecksStateItem> CHECKS_STATE_ITEM_COMPARATOR =
            Comparator.comparing(JugglerChecksStateItem::getService);

    private final Session client;
    private final JugglerClient jugglerClient;
    private final YpClient ypClient;
    private List<JugglerChecksStateItem> failedRegressionChecks;
    private final TsShardsProvider shardsProvider;


    public JobsTestReport(
            @Qualifier(MAINTENANCE_HELPERS_TRACKER_BEAN) Session trackerClient,
            JugglerClient jugglerClient,
            YpClient ypClient,
            TsShardsProviderFactory tsShardsProviderFactory
    ) {
        this.client = trackerClient;
        this.jugglerClient = jugglerClient;
        this.ypClient = ypClient;
        this.shardsProvider = tsShardsProviderFactory.readOnly();
        this.failedRegressionChecks = Collections.emptyList();
    }

    @Override
    public void execute() {
        logger.debug("'jobs test report' task started");
        List<String> hosts = ypClient.getHosts(JOBS_DEPLOY_STAGE_NAME);

        logger.debug("finding release ticket");
        String query = "Queue: " + QUEUE +
                " AND Type: Release" +
                " AND Components: " + StartrekComponent.JAVA_JOBS +
                " AND Components: " + StartrekComponent.RELEASES_JAVA_DIRECT +
                " AND Status: Testing, \"RM Acceptance\"" +
                " AND Tags: " + StartrekTag.CHECK_JOBS_REGRESSION;
        client.issues().find(query).forEachRemaining(ticket -> report(ticket, hosts));

        failedRegressionChecks.clear();
        logger.debug("'jobs test report' task completed");
    }

    private void report(Issue ticket, List<String> hosts) {
        logger.info("release ticket founded: {}", ticket);
        String comment;
        try {
            StringBuilder sb = new StringBuilder("Проверено на ТС по чеклисту v5.7:\n");
            sb.append(getSchedulerLogInfo()).append('\n');
            sb.append(getStartedGraphInfo()).append('\n');
            sb.append("===Регрессия\n");
            sb.append(getJugglerChecksInfo());
            if (!failedRegressionChecks.isEmpty()) {
                sb.append(createTicketsForFailedJobs(hosts, ticket)).append('\n');
            }
            sb.append(getCampAggregatedLastChangeProcessorInfo()).append('\n');
            sb.append(getSwitchmanStatCollectorInfo()).append('\n');
            logger.debug("comment:\n{}", sb);
            comment = sb.toString();
        } catch (Exception e) {
            logger.error("An error occurred while creating the report: ", e);
            comment = "Не удалось сформировать релизный отчет: " + e;
        }
        logger.info("remove '{}' from tags", StartrekTag.CHECK_JOBS_REGRESSION);
        IssueUpdate.Builder updateBuilder = IssueUpdate.builder().removeTags(StartrekTag.CHECK_JOBS_REGRESSION);
        updateBuilder.comment(signedMessageFactory(this, SELF, comment).build());
        ticket.update(updateBuilder.build());
    }

    private StringBuilder getSchedulerLogInfo() throws URISyntaxException {
        StringBuilder sb = new StringBuilder();
        sb.append("===Hourglass scheduler\n");
        sb.append(getJugglerSchedulerInfo());
        sb.append(getSuccessesGraphInfo());
        return sb;
    }

    private StringBuilder getSuccessesGraphInfo() throws URISyntaxException {
        String dateFrom = DateTimeFormatter.ISO_DATE_TIME.format(LocalDateTime.now(Clock.systemUTC()).minusDays(2));
        String dateTo = DateTimeFormatter.ISO_DATE_TIME.format(LocalDateTime.now(Clock.systemUTC()));

        URIBuilder bld = new URIBuilder(SOLOMON_URL);
        bld.addParameter("project", "direct-test");
        bld.addParameter("cluster", "app_java-jobs");
        bld.addParameter("service", "java-monitoring");
        bld.addParameter("l.host", "!CLUSTER");
        bld.addParameter("l.sensor", "quartz_successes");
        bld.addParameter("l.env", "testing");
        bld.addParameter("graph", "auto");
        bld.addParameter("b", dateFrom + "Z");
        bld.addParameter("e", dateTo + "Z");
        bld.addParameter("hideNoData", "true");
        bld.addParameter("graphOnly", "y");
        bld.addParameter("autorefresh", "n");

        StringBuilder sb = new StringBuilder("* График количества успешно выполненных заданий //(на нем должна быть " +
                "ступенька в ноль в момент выкладки)//:\n" +
                "  Убедитесь, что перезапустились **ВСЕ** инстансы. Eсли по графику непонятно - проверяйте логи.\n" +
                "{{iframe height='150' width='100%' src='");
        sb.append(bld.build());
        sb.append("' frameborder='0'}}");
        return sb;
    }

    private StringBuilder getStartedGraphInfo() throws URISyntaxException {
        String dateFrom = DateTimeFormatter.ISO_DATE_TIME.format(LocalDateTime.now(Clock.systemUTC()).minusHours(8));
        String dateTo = DateTimeFormatter.ISO_DATE_TIME.format(LocalDateTime.now(Clock.systemUTC()));

        URIBuilder bld = new URIBuilder(SOLOMON_URL);
        bld.addParameter("project", "direct-test");
        bld.addParameter("cluster", "app_java-jobs");
        bld.addParameter("service", "java-monitoring");
        bld.addParameter("l.host", "!CLUSTER");
        bld.addParameter("l.sensor", "quartz_started");
        bld.addParameter("l.env", "testing");
        bld.addParameter("graph", "auto");
        bld.addParameter("transform", "differentiate");
        bld.addParameter("b", dateFrom + "Z");
        bld.addParameter("e", dateTo + "Z");
        bld.addParameter("hideNoData", "true");
        bld.addParameter("graphOnly", "y");
        bld.addParameter("autorefresh", "n");

        StringBuilder sb = new StringBuilder("* График количества запускаемых задач, в секунду //(должен быть " +
                "ровным, без провалов)//:\n{{iframe height='250' width='100%' src='");
        sb.append(bld.build());
        sb.append("' frameborder='0'}}");
        return sb;
    }

    private StringBuilder createTicketsForFailedJobs(List<String> hosts, Issue ticket) throws URISyntaxException {
        if (failedRegressionChecks.isEmpty()) {
            return null;
        }
        StringBuilder sb = new StringBuilder();
        sb.append(LINE_SEPARATOR);
        sb.append("\nНа ТС обнаружена проблема: \n");

        String nowStr = formatDateTimeForLink(LocalDateTime.now());
        String prevStr = formatDateTimeForLink(LocalDateTime.now().minusDays(2));
        String hostsStr = String.join(" ", hosts);
        String tagRelease = ticket.getKey().toLowerCase();

        String prefix = "jobs.";
        String suffix = ".working.test";
        for (JugglerChecksStateItem item : failedRegressionChecks) {
            String tagIdentity = item.getService() + "-" + item.getStatus();
            Issue failedRegressionTicket = getIssue(List.of(
                    StartrekTag.FOUND_IN_JOBS_REGRESSION, StartrekTag.CREATED_BY_CHECKER, tagIdentity, tagRelease));
            if (failedRegressionTicket != null) {
                sb.append(failedRegressionTicket.getKey()).append("\n");
                continue;
            }

            String job = item.getService();
            job = job.startsWith(prefix) ? job.substring(prefix.length()) : job;
            job = job.endsWith(suffix) ? job.substring(0, job.length() - suffix.length()) : job;
            StringBuilder logLink = new StringBuilder();
            if (!job.isEmpty()) {
                StringBuilder sbFragment = new StringBuilder("messages,");
                sbFragment.append(prevStr).append(',');
                sbFragment.append(nowStr).append(',');
                sbFragment.append("false,1:101,3,log_time_nanos,trace_id,parent_id,host,");
                sbFragment.append(hostsStr).append(',');
                sbFragment.append("method,%.").append(job);

                URIBuilder bld = new URIBuilder("https://test-direct.yandex.ru/logviewer/");
                bld.setFragment(sbFragment.toString());

                logLink.append("((");
                logLink.append(bld.build());
                logLink.append(" логи джобы за последние два дня))");
            }
            URIBuilder bld = new URIBuilder("https://juggler.yandex-team.ru/check_details/");
            bld.addParameter("host", item.getHost());
            bld.addParameter("service", item.getService());

            logger.info("create new ticket");
            Issue issue = client.issues().create(IssueCreate.builder()
                    .queue(QUEUE)
                    .tags(StartrekTag.FOUND_IN_JOBS_REGRESSION, StartrekTag.CREATED_BY_CHECKER, tagIdentity, tagRelease)
                    .priority(2)
                    .type(1)
                    .summary("Горит мониторинг на ТС: " + item.getService() + " - " + item.getStatus())
                    .description("((" + bld.build() + " мониторинг))\n" + logLink +
                            "\n\nНужно выяснить причину падения, определить ее критичность к релизу, починить")
                    .build());
            logger.info("ticket created: {}", issue);

            sb.append(issue.getKey()).append("\n");
        }
        sb.append(LINE_SEPARATOR);
        return sb;
    }

    private StringBuilder getJugglerSchedulerInfo() throws URISyntaxException {

        URIBuilder bld = new URIBuilder("https://juggler.yandex-team.ru/aggregate_checks/");
        bld.addParameter("query", "namespace=direct.test&tag=direct_jobs_scheduler");

        StringBuilder sb = new StringBuilder("* ((");
        sb.append(bld.build());
        sb.append(" Juggler-мониторинг)) //(шедулер работает на всех хостах, ");
        sb.append("запускает задачи, запущен достаточно давно)//:\n");

        JugglerChecksStateFilter filter = new JugglerChecksStateFilter(JUGGLER_HOST_DIRECT_CHECKS_TEST,
                "direct_jobs_scheduler");
        List<JugglerChecksStateItem> jugglerItems = jugglerClient.getChecksStateItems(filter);
        jugglerItems.sort(CHECKS_STATE_ITEM_COMPARATOR);

        SimpleDateFormat formatter = new SimpleDateFormat("EEE MMM dd HH:mm:ss yyyy");
        List<String> headers = List.of("Host", "Service", "Status", "Description", "Status changed at");
        List<Function<JugglerChecksStateItem, Object>> columns =
                List.of(JugglerChecksStateItem::getHost,
                        j -> getServiceTitle(j.getHost(), j.getService()),
                        j -> getStatusTitle(j.getStatus()),
                        JugglerChecksStateItem::getDescription,
                        j -> formatter.format(new Date(j.getChangeTime().longValue() * 1000L)));
        sb.append(buildTable(headers, jugglerItems, columns));
        return sb;
    }

    private String getStatusTitle(String status) {
        switch (status) {
            case "OK":
                return "!!(green)OK!!";
            case "WARN":
                return "!!(yellow)WARN!!";
            case "CRIT":
                return "!!(red)CRIT!!";
            default:
                return status;
        }
    }

    private String getServiceTitle(String host, String service) {
        try {
            URIBuilder bld = new URIBuilder("https://juggler.yandex-team.ru/check_details/");
            bld.addParameter("host", host);
            bld.addParameter("service", service);
            return "((" + bld.build() + " " + service + "))";
        } catch (URISyntaxException e) {
            return e.toString();
        }
    }

    private StringBuilder getJugglerChecksInfo() throws URISyntaxException {

        URIBuilder bld = new URIBuilder("https://juggler.yandex-team.ru/aggregate_checks/");
        bld.addParameter("query", "namespace=direct.test&tag=direct_jobs_release_regression");

        StringBuilder sb = new StringBuilder("* ((");
        sb.append(bld.build());
        sb.append(" Мониторинг задач)) //(задачи запускаются и не падают, а значит их проверки должны быть OK)// в <{Juggler\n");

        JugglerChecksStateFilter filter = new JugglerChecksStateFilter(JUGGLER_HOST_DIRECT_CHECKS_TEST,
                "direct_jobs_release_regression");
        List<JugglerChecksStateItem> jugglerItems = jugglerClient.getChecksStateItems(filter);
        jugglerItems.sort(CHECKS_STATE_ITEM_COMPARATOR);

        SimpleDateFormat formatter = new SimpleDateFormat("EEE MMM dd HH:mm:ss yyyy");
        List<String> headers = List.of("Host", "Service", "Status", "Status changed at");
        List<Function<JugglerChecksStateItem, Object>> columns =
                List.of(JugglerChecksStateItem::getHost,
                        j -> getServiceTitle(j.getHost(), j.getService()),
                        j -> getStatusTitle(j.getStatus()),
                        j -> formatter.format(new Date(j.getChangeTime().longValue() * 1000L)));
        sb.append(buildTable(headers, jugglerItems, columns));

        failedRegressionChecks = jugglerItems.stream()
                .filter(i -> i.getStatus().equalsIgnoreCase("WARN") || i.getStatus().equalsIgnoreCase("CRIT"))
                .collect(Collectors.toList());
        logger.info("found {} failed schedulers: {}", failedRegressionChecks.size(), failedRegressionChecks);
        sb.append("}>\n");
        return sb;
    }

    private StringBuilder getSwitchmanStatCollectorInfo() throws URISyntaxException {
        String dateFrom = DateTimeFormatter.ISO_DATE_TIME.format(LocalDateTime.now(Clock.systemUTC()).minusDays(2));
        String dateTo = DateTimeFormatter.ISO_DATE_TIME.format(LocalDateTime.now(Clock.systemUTC()));

        URIBuilder bld = new URIBuilder(SOLOMON_URL);
        bld.addParameter("project", "direct-test");
        bld.addParameter("cluster", "app_java-jobs-push");
        bld.addParameter("service", "push-monitoring");
        bld.addParameter("l.host", "CLUSTER");
        bld.addParameter("l.sensor", "locks_count");
        bld.addParameter("l.env", "testing");
        bld.addParameter("graph", "auto");
        bld.addParameter("b", dateFrom + "Z");
        bld.addParameter("e", dateTo + "Z");
        bld.addParameter("graphOnly", "y");
        bld.addParameter("autorefresh", "n");

        StringBuilder sb = new StringBuilder("* SwitchmanStatCollector //(график не должен быть пустым)//" +
                "<{работает:\n{{iframe height='308' width='100%' src='");
        sb.append(bld.build());
        sb.append("' frameborder='0'}}");
        sb.append("\n}>");
        return sb;
    }

    private Issue getIssue(Collection<String> tags) {
        String query = "Queue: " + QUEUE +
                " AND Tags: " + String.join(" AND Tags: ", tags);
        IteratorF<Issue> issues = client.issues().find(query);
        return issues.hasNext() ? issues.next() : null;
    }

    private StringBuilder getCampAggregatedLastChangeProcessorInfo() {
        StringBuilder sb = new StringBuilder();
        sb.append("* CampAggregatedLastсhangeProcessor //(большинство дат должны быть свежими)// <{работает\n");
        sb.append("%%select max(`last_change`)\n");
        sb.append("from `camp_aggregated_lastchange`%%\n");

        DateTimeFormatter formatter = DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm:ss");
        List<Pair<String, String>> values = new ArrayList<>();

        shardsProvider.get().forEach(dbWrapper -> {
            DSLContext dslContext = dbWrapper.getDslContext();
            LocalDateTime dateTime = dslContext
                    .select(DSL.max(Tables.CAMP_AGGREGATED_LASTCHANGE.LAST_CHANGE).as("max(last_change)"))
                    .from(Tables.CAMP_AGGREGATED_LASTCHANGE)
                    .fetchOne()
                    .value1();
            String dt = dateTime != null ? dateTime.format(formatter) : "шард пуст";
            var shard = dbWrapper.getDbname().split(":")[1];
            values.add(Pair.of(shard, dt));
        });

        List<String> headers = List.of("Shard", "Last change");
        sb.append(buildTable(headers, values, List.of(Pair::getKey, Pair::getValue)));
        sb.append("}>");
        return sb;
    }

    private String formatDateTimeForLink(LocalDateTime date) {
        return date.format(DateTimeFormatter.ofPattern("yyyyMMdd HHmmss")).replace(' ', 'T');
    }

    private static class DbConfigPatcher {
        private final LiveResource password;

        public DbConfigPatcher(LiveResource password) {
            this.password = password;
        }

        String patchJson(String json) {
            JsonNode rootJson = fromJson(json);
            JsonNode jsonNode = rootJson.get("db_config");

            if (jsonNode.has("user")) {
                ((ObjectNode) jsonNode).put("user", "direct-ro");
            }

            if (jsonNode.has("pass")) {
                ((ObjectNode) jsonNode).put("pass", password.getContent().trim());
            }

            if (jsonNode.has("extra_users")) {
                ((ObjectNode) jsonNode).putObject("extra_users");
            }

            return toJson(rootJson);
        }
    }

    private static class ListenerProxy implements LiveResourceListener {
        private final LiveResourceListener originalListener;
        private final DbConfigPatcher patcher;

        public ListenerProxy(LiveResourceListener originalListener, DbConfigPatcher patcher) {
            this.originalListener = originalListener;
            this.patcher = patcher;
        }


        @Override
        public void update(LiveResourceEvent event) {
            String currentContent = event.getCurrentContent();
            String patchedContent = patcher.patchJson(currentContent);
            LiveResourceEvent patchedEvent = new LiveResourceEvent(patchedContent);
            originalListener.update(patchedEvent);
        }
    }
}
