package ru.yandex.direct.chassis.entity.startrek

import org.slf4j.LoggerFactory
import ru.yandex.direct.chassis.util.DirectAppsConfProvider
import ru.yandex.direct.chassis.properties.PropertiesLightSupport
import ru.yandex.direct.chassis.util.DirectReleaseUtils
import ru.yandex.direct.chassis.util.startrek.StartrekHelper
import ru.yandex.direct.chassis.util.TabulaUtils.timelineLink
import ru.yandex.direct.chassis.util.startrek.StartrekStatusKey
import ru.yandex.direct.env.ProductionOnly
import ru.yandex.direct.scheduler.Hourglass
import ru.yandex.direct.scheduler.support.DirectJob
import ru.yandex.startrek.client.model.CommentUpdate
import ru.yandex.startrek.client.model.Issue
import java.time.LocalDateTime
import java.time.format.DateTimeFormatter
import kotlin.text.Regex.Companion.escape

/**
 * Комментарий про главное в релизе
 */
private const val OVERVIEW_AUTOCOMMENT = "====<# <a name=\"overview\">Главное про релиз</a> #>=="

/**
 * С каких слов начинается автообновляемый комментарий, не учитывая шапку комментария
 * Нужен, чтобы определять с какого места перезаписывать комментарий при следующем обновлении
 */
private const val OVERVIEW_AUTOCOMMENT_STARTS_WITH = "Состояние на"

private const val OVERVIEW_AUTOCOMMENT_RELEASES_ENABLED = "overview_autocomment_releases_enabled"

/**
 * Маркер, между которыми находится полезное содержимое коментария
 */
private const val OVERVIEW_AUTOCOMMENT_CONTENT_MARKER = "%%(comments) content%%"

/**
 * Выражение, которое матчит содержимое комментария (текст между двумя маркерами)
 */
private val OVERVIEW_AUTOCOMMENT_CONTENT_REGEXP =
    "${escape(OVERVIEW_AUTOCOMMENT_CONTENT_MARKER)}(.*)${escape(OVERVIEW_AUTOCOMMENT_CONTENT_MARKER)}"
        .toRegex(RegexOption.DOT_MATCHES_ALL)

/**
 * Автообновляемый комментарий "Главное в релизе"
 *
 * См. подробное описание в [DIRECT-135031](https://st.yandex-team.ru/DIRECT-135031)
 */
@Hourglass(periodInSeconds = 2 * 60, needSchedule = ProductionOnly::class)
class OverviewAutocommentReleasesJob(
    private val startrek: StartrekHelper,
    private val propertiesLightSupport: PropertiesLightSupport,
    private val directAppsConfProvider: DirectAppsConfProvider,
) : DirectJob() {

    companion object {
        private val logger = LoggerFactory.getLogger(OverviewAutocommentReleasesJob::class.java)
        private val LAST_UPDATE_DATETIME_FORMATTER = DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm:ss")
    }

    private val isProcessorEnabled: Boolean
        get() {
            // По умолчанию джоба включена
            val value = propertiesLightSupport[OVERVIEW_AUTOCOMMENT_RELEASES_ENABLED] ?: return true
            return try {
                value.toInt() == 1
            } catch (e: NumberFormatException) {
                logger.error("Failed to parse $OVERVIEW_AUTOCOMMENT_RELEASES_ENABLED = $value")
                false
            }
        }

    override fun execute() {
        if (!isProcessorEnabled) {
            logger.info("Skip processing. OverviewAutocommentReleasesJob is not enabled")
            return
        }

        val component2app = directAppsConfProvider
            .getDirectAppsConf()
            .filter { it.trackerComponent.isNotBlank() }
            .associateBy { it.trackerComponent }

        val releases = startrek.session.issues().find(
            """
            | Queue: DIRECT
            | Type: Release
            | Components: "Releases: JavaDirect"
            | Status: Testing, ReadyForTest, "RM Acceptance"
            | "Sort by": key desc
            """.trimMargin()
        ).toList()

        for (release in releases) {
            val app = release.components.asIterable()
                .map { component2app[it.display] }
                .firstOrNull { it != null }
                ?: continue

            val affectedApp = app.trackerAffectedApp
                ?: continue

            logger.info("Process issue ${release.key}")
            val oldOverviewComment = startrek.session.comments().getAll(release)
                .find { it.text.isMatch { t -> t.contains(OVERVIEW_AUTOCOMMENT) } }
                .orNull

            if (oldOverviewComment == null) {
                logger.info("Create autocomment for issue ${release.key}")
                val newOverviewText = buildNewOverviewComment(release, affectedApp, null)
                startrek.addComment(release, comment = newOverviewText, notify = false, footer = false)
            } else {
                val oldOverviewText = oldOverviewComment.text.get()
                val newOverviewText = buildNewOverviewComment(release, affectedApp, oldOverviewText)

                if (newOverviewText != oldOverviewText) {
                    val commentUpdate = CommentUpdate.builder().comment(newOverviewText).build()
                    startrek.session.comments().update(release, oldOverviewComment, commentUpdate, false, false)
                    logger.info("Updated overview comment for issue ${release.key}")
                } else {
                    logger.info("Content is the same")
                }
            }
        }
    }

    /**
     * Строит содержимое комментария. При совпадении "полезного содержимого" (часть комментария между двумя маркерами)
     * возвращает точно совпадающий со старым текст. Это важно, чтобы не обновлять комментарий при каждом запуске джобы
     */
    private fun buildNewOverviewComment(release: Issue, trackerAppName: String, oldCommentText: String?): String {
        val newContent = buildNewOverview(release, trackerAppName)

        return if (oldCommentText != null) {
            val (header, oldTextWithoutHeader) = splitHeader(oldCommentText)
            val oldContent = extractContent(oldTextWithoutHeader)

            header + if (oldContent == newContent) {
                oldTextWithoutHeader
            } else {
                wrapAutocommentContent(newContent)
            }
        } else {
            """
                |$OVERVIEW_AUTOCOMMENT
                |${wrapAutocommentContent(newContent)}
            """.trimMargin()
        }
    }

    /**
     * Построить новое полезное содержимое комментария (без маркеров и заголовка)
     */
    private fun buildNewOverview(release: Issue, trackerAppName: String): String {
        fun createIssueLine(title: String, query: String): String {
            val count = startrek.issuesCount(query)
            val link = StartrekHelper.filterLink(query, startrek.startrekUrl)
            return "$title ($count): (($link ссылка на запрос в Трекере))"
        }

        fun createIssueLineWithoutCount(title: String, query: String): String {
            val link = StartrekHelper.filterLink(query, startrek.startrekUrl)
            return "$title: (($link ссылка на запрос в Трекере))"
        }

        return """
            |${createIssueLine("Тестировать", toTestQuery(trackerAppName, release.key))}
            |${createIssueLine("Разобраться", toReviewQuery(trackerAppName, release.key))}
            |
            |${computeTestIssuesState(release)}
            |
            |${createIssueLine("Все сделано", allOkQuery(trackerAppName, release.key))}
            |${createIssueLine("Не предполагалось тестировать", wontTestQuery(release.key))}
            |${createIssueLineWithoutCount("Задачи из соседних очередей, попавшие в релиз", toExternalReviewQuery(trackerAppName, release.key))}
            |
            |Справка по релизам: https://docs.yandex-team.ru/direct-dev/guide/releases/release-newci
            |
            |Таймлайн релиза: ((${timelineLink(release.key)} ссылка на таймлайн))
        """.trimMargin()
    }

    /**
     * Возвращает полезное содержимое комментария, правильно обернутое маркерами
     */
    private fun wrapAutocommentContent(
        content: String,
        time: LocalDateTime = LocalDateTime.now(),
    ) = """
        |$OVERVIEW_AUTOCOMMENT_STARTS_WITH ${LAST_UPDATE_DATETIME_FORMATTER.format(time)}
        |$OVERVIEW_AUTOCOMMENT_CONTENT_MARKER
        |$content
        |$OVERVIEW_AUTOCOMMENT_CONTENT_MARKER
        |${startrek.createFooter()}
    """.trimMargin()

    /**
     * Достает из текста комментария полезное содержимое
     * (текст между маркерами [OVERVIEW_AUTOCOMMENT_CONTENT_MARKER], либо `null`)
     */
    private fun extractContent(commentText: String): String? {
        val result = OVERVIEW_AUTOCOMMENT_CONTENT_REGEXP.find(commentText)
        val content = result?.groupValues?.get(1)
        return content?.trim()
    }

    /**
     * Отделяет изначальный заголовок (текст _Главное в релизе_), от остального содержимого комментария.
     * Рассчитывает, что содержимое начинается на `OVERVIEW_AUTOCOMMENT_STARTS_WITH`
     */
    private fun splitHeader(commentText: String): Pair<String, String> {
        val stateStartIndex = commentText.indexOf(OVERVIEW_AUTOCOMMENT_STARTS_WITH)
        return if (stateStartIndex == -1) {
            commentText + "\n" to ""
        } else {
            commentText.substring(0, stateStartIndex) to commentText.substring(stateStartIndex)
        }
    }

    private fun computeTestIssuesState(release: Issue): String {
        try {
            val unitTestIssues = DirectReleaseUtils.findUnitTestIssues(release)
            logger.info("Found ${unitTestIssues.size} unit test issues for issue ${release.key}")

            val regressionIssues = DirectReleaseUtils.findRegressionIssues(release)
            logger.info("Found ${regressionIssues.size} regression issues for issue ${release.key}")

            val unitTestIssuesState = getIssuesState(unitTestIssues, "Юнит-тесты")
            val regressionIssuesState = getIssuesState(regressionIssues, "Регрессия")

            return """
            $unitTestIssuesState
            $regressionIssuesState
            """.trimIndent()
        } catch (e: RuntimeException) {
            logger.error("Failed to get test issues state", e)
            return ""
        }
    }

    private fun getIssuesState(issues: List<Issue>, name: String): String {
        if (issues.isEmpty()) {
            return ""
        }
        val isAnyNotClosedIssue = issues.any { it.status.key != StartrekStatusKey.CLOSED }
        val issueKeys = issues.joinToString { it.key }
        return if (isAnyNotClosedIssue) {
            "$name: в процессе $issueKeys"
        } else {
            "$name: done $issueKeys"
        }
    }
}
