package ru.yandex.direct.chassis.entity.startrek

import org.springframework.stereotype.Component
import ru.yandex.direct.chassis.util.DirectAppsConfEntry
import ru.yandex.direct.chassis.util.DirectAppsConfProvider
import ru.yandex.direct.chassis.util.DirectReleaseUtils
import ru.yandex.direct.chassis.util.nonCf
import ru.yandex.direct.chassis.util.startrek.StartrekComponent
import ru.yandex.direct.chassis.util.startrek.StartrekHelper
import ru.yandex.direct.chassis.util.startrek.StartrekQueue
import ru.yandex.direct.chassis.util.startrek.StartrekStatusKey
import ru.yandex.startrek.client.model.Issue


@Component
class ReleaseDependencyService(
    private val startrek: StartrekHelper,
    private val directAppsConfProvider: DirectAppsConfProvider,
) {

    private val appsByName: Map<String, DirectAppsConfEntry>
        get() = directAppsConfProvider.getDirectAppsConf()
            .associateBy { it.name }

    /**
     * Проверяет, что релиз [release] можно выкладывать (нету неудовлетворенных зависимостей)
     */
    fun checkReleaseDependenciesSatisfied(graph: DependencyGraph, release: Issue): Boolean {
        // Проверить, что среди существующих зависимостей нету циклов
        if (graph.findCycle() != null) {
            return false
        }

        val app = DirectReleaseUtils.getReleaseApp(release, appsByName.values)?.name
            ?: return false
        val ticketKeys = DirectReleaseUtils.getReleaseTickets(release)

        // Для каждого тикета в релизе, проверить что зависимость удовлетворена
        val nodes = ticketKeys.map { ticket -> DependencyNode(app, ticket) }
        val unsatisfiedDependencies = graph.findUnsatisfiedDependencies(nodes)

        return unsatisfiedDependencies.isEmpty()
    }

    fun buildDependencyGraph(): DependencyGraph {
        val dependencies = findOpenDependencies()
        val parsedDependencies = dependencies
            .map { parseDependency(it) }
            .filter { it.ok }

        val edgeList = parsedDependencies
            .filter { it.ok }
            .flatMap { it.edges }

        val issueKeys = edgeList
            .flatMap { listOf(it.from.ticket, it.to.ticket) }
            .toSet()
        val issues: Map<String, Issue> = startrek.findIssuesByKeys(issueKeys)
            .associateBy { it.key }

        // Пропускаем зависимости, ссылающиеся на несуществующие тикеты
        val filteredEdgeList = edgeList
            .filter { it.from.ticket in issues }
            .filter { it.to.ticket in issues }

        val edges: Map<DependencyNode, List<DependencyNode>> = filteredEdgeList
            .groupBy({ it.from }, { it.to })
        val reverseEdges: Map<DependencyNode, List<DependencyNode>> = filteredEdgeList
            .groupBy({ it.to }, { it.from })

        return DependencyGraph(appsByName, issues, edges, reverseEdges)
    }

    /**
     * Парсит один тикет-зависимость. Не проверяет существование тикетов, на которые ссылаются зависимости
     */
    private fun parseDependency(dependency: Issue): DependencyResult {
        val edges = mutableListOf<DependencyEdge>()
        val errors = mutableListOf<DependencyError>()
        val warnings = mutableListOf<DependencyError>()

        if (StartrekComponent.RELEASE_DEPENDENCY !in
            dependency.components.nonCf().map { it.display }
        ) {
            errors += DependencyError(DependencyErrorType.NO_RELEASE_COMPONENT)
        }

        if (!dependency.summary.startsWith("Зависимость релизов:")) {
            warnings += DependencyError(DependencyErrorType.UNEXPECTED_SUMMARY)
        }

        var inSection = false
        var prevNode: DependencyNode? = null
        var currentNode: DependencyNode? = null

        for (line in dependency.description.orElse("").lines()) {
            if (line.contains("""^(\*\*)?Порядок""".toRegex(RegexOption.IGNORE_CASE))) {
                if (inSection && currentNode == null) {
                    warnings += DependencyError(DependencyErrorType.EMPTY_SECTION)
                }
                inSection = true
                prevNode = null
                currentNode = null
                continue
            }

            val match = """^([^/]*)/(.*)$""".toRegex().matchEntire(line)
            if (inSection && match != null) {
                prevNode = currentNode
                currentNode = DependencyNode(
                    app = match.groupValues[1].trim(),
                    ticket = match.groupValues[2].trim(),
                )

                val appEntry = appsByName[currentNode.app]
                if (appEntry == null) {
                    errors += DependencyError(DependencyErrorType.INVALID_APP_NAME, currentNode.app)
                } else if (appEntry.trackerDeployedTag == null) {
                    errors += DependencyError(DependencyErrorType.UNSUPPORTED_APP, currentNode.app)
                }

                if (prevNode != null) {
                    edges += prevNode to currentNode
                }

                continue
            }

            if (inSection) {
                if (currentNode == null) {
                    warnings += DependencyError(DependencyErrorType.EMPTY_SECTION)
                } else if (prevNode == null) {
                    warnings += DependencyError(DependencyErrorType.SINGLE_NODE_SECTION)
                }
                inSection = false
                currentNode = null
                prevNode = null
            }
        }

        return DependencyResult(edges, errors, warnings)
    }

    private fun findOpenDependencies(): List<Issue> {
        return startrek.findIssues("""
            Queue: ${StartrekQueue.DIRECT}
            Components: "${StartrekComponent.RELEASE_DEPENDENCY}"
            Status: !${StartrekStatusKey.CLOSED}
            "Sort by": key desc
        """.trimIndent())
    }

}

class DependencyGraph(
    private val apps: Map<String, DirectAppsConfEntry>,
    private val issues: Map<String, Issue>,
    private val edges: Map<DependencyNode, List<DependencyNode>>,
    private val reverseEdges: Map<DependencyNode, List<DependencyNode>>,
) {
    fun findCycle(): List<DependencyNode>? {
        val color: MutableMap<DependencyNode, Int> = mutableMapOf()
        val parent: MutableMap<DependencyNode, DependencyNode> = mutableMapOf()

        fun markNodes(node: DependencyNode): DependencyEdge? {
            color[node] = 1

            for (child in edges[node].orEmpty()) {
                when (color[child]) {
                    1 -> return DependencyEdge(node, child)
                    2 -> continue
                }

                parent[child] = node
                val cycleEdge = markNodes(child)
                if (cycleEdge != null) {
                    return cycleEdge
                }
            }

            color[node] = 2
            return null
        }

        for (node in edges.keys) {
            if (node !in color) {
                val cycleEdge = markNodes(node)
                    ?: continue

                val cycle = mutableListOf<DependencyNode>()
                var current = cycleEdge.from
                while (current != cycleEdge.to) {
                    cycle.add(current)
                    current = parent[current]!!
                }
                cycle.add(current)

                return cycle.reversed()
            }
        }

        return null
    }

    fun findUnsatisfiedDependencies(nodes: List<DependencyNode>): Set<DependencyNode> {
        val dependencies = nodes.flatMap { node -> findUnsatisfiedDependencies(node) }.toSet()
        return dependencies - nodes.toSet()
    }

    private fun findUnsatisfiedDependencies(node: DependencyNode): Set<DependencyNode> {
        return reverseEdges[node].orEmpty()
            .filter { parent -> !isDeployed(parent) }
            .flatMap { parent -> setOf(parent) + findUnsatisfiedDependencies(parent) }
            .toSet()
    }

    private fun isDeployed(node: DependencyNode): Boolean {
        val tag = apps[node.app]?.trackerDeployedTag ?: return false
        return tag in issues[node.ticket]!!.tags
    }
}

data class DependencyResult(
    val edges: List<DependencyEdge>,
    val errors: List<DependencyError>,
    val warnings: List<DependencyError>,
) {
    val ok = errors.isEmpty()
}

data class DependencyEdge(
    val from: DependencyNode,
    val to: DependencyNode,
)

data class DependencyNode(
    val app: String,
    val ticket: String,
) {
    infix fun to(other: DependencyNode): DependencyEdge {
        return DependencyEdge(this, other)
    }
}

data class DependencyError(
    val type: DependencyErrorType,
    val value: Any? = null,
)

enum class DependencyErrorType {
    // dependency parsing errors
    NO_RELEASE_COMPONENT,
    UNEXPECTED_SUMMARY,
    EMPTY_SECTION,
    SINGLE_NODE_SECTION,
    INVALID_APP_NAME,
    UNSUPPORTED_APP,
}
