package ru.yandex.direct.chassis.entity.startrek

import org.slf4j.LoggerFactory
import ru.yandex.direct.chassis.properties.PropertiesLightSupport
import ru.yandex.direct.chassis.util.startrek.StartrekField
import ru.yandex.direct.chassis.util.startrek.StartrekHelper
import ru.yandex.direct.chassis.util.startrek.StartrekHelper.Companion.field
import ru.yandex.direct.chassis.util.startrek.StartrekHelper.Companion.issueUpdate
import ru.yandex.direct.env.ProductionOnly
import ru.yandex.direct.scheduler.Hourglass
import ru.yandex.direct.scheduler.support.DirectJob
import ru.yandex.startrek.client.model.Issue
import java.time.Duration
import java.time.Instant

/**
 * Джоба для обновления поля "Количество жалоб" при изменении полей
 * "Ожидаемое количество пользователей" и "Тикеты или заявки в Space"
 */
@Hourglass(periodInSeconds = 5 * 60, needSchedule = ProductionOnly::class)
class UpdateNumberOfComplaintsJob(
    private val startrek: StartrekHelper,
    private val properties: PropertiesLightSupport,
) : DirectJob() {

    override fun execute() {
        val startedProcessing = Instant.now()

        val secondsFromLastUpdate = lastUpdateTime()
            ?.let { startedProcessing.epochSecond - it.epochSecond }
            ?.let { it + 60 } // возьмём на минуту больше (на всякий случай)
            ?: DEFAULT_TIME_DELTA

        startrek.session.issues()
            .find("""
                Queue: DIRECTSUP
                Updated: >= now() - ${secondsFromLastUpdate}s
                "Sort by": Updated desc""".trimIndent())
            .toList()
            .also { logger.info("Found ${it.size} issues") }
            .forEach { it.updateNumbersOfComplaints() }

        properties[LAST_UPDATED_TIME] = startedProcessing.epochSecond.toString()
        val processingTime = Instant.now().toEpochMilli() - startedProcessing.toEpochMilli()
        logger.info("Processed in $processingTime ms")
    }

    private fun Issue.updateNumbersOfComplaints() {
        val complaints = numberOfComplaints() ?: 0
        val newComplaints = numberOfCrmTickets() + (numberOfUsers() ?: 0)
        if (complaints != newComplaints) {
            val update = issueUpdate {
                set(StartrekField.NUMBER_OF_COMPLAINTS, newComplaints)
            }
            logger.info("https://st.yandex-team.ru/$key: " +
                "updated `numberOfComplaints` from $complaints to $newComplaints")
            startrek.session.issues().update(this, update, false, false)
        } else {
            logger.info("https://st.yandex-team.ru/$key: " +
                "`numberOfComplaints` is $complaints, changed nothing")
        }
    }

    private fun lastUpdateTime(): Instant? {
        val secondsString = properties[LAST_UPDATED_TIME] ?: return null
        val seconds = secondsString.toLongOrNull() ?: run {
            logger.error("Can't parse seconds from the epoch: $secondsString")
            return null
        }
        return Instant.ofEpochSecond(seconds)
    }

    companion object {
        private val logger = LoggerFactory.getLogger(UpdateNumberOfComplaintsJob::class.java)
        private val DEFAULT_TIME_DELTA = Duration.ofDays(14).seconds
        private const val LAST_UPDATED_TIME = "last_updated_complaints_time"

        private fun Issue.numberOfCrmTickets(): Int =
            getO<List<*>>(StartrekField.NUMBER_OF_CRM_TICKETS).orNull?.size ?: 0

        private fun Issue.numberOfUsers(): Long? = field(StartrekField.NUMBER_OF_USERS)

        private fun Issue.numberOfComplaints(): Long? = field(StartrekField.NUMBER_OF_COMPLAINTS)
    }
}
