package ru.yandex.direct.chassis.entity.telegram

import com.google.re2j.Pattern
import org.slf4j.LoggerFactory
import org.springframework.stereotype.Component
import ru.yandex.direct.chassis.entity.deploy.DirectReleasesService
import ru.yandex.direct.chassis.entity.telegram.TelegramUtils.htmlLink
import ru.yandex.direct.chassis.entity.telegram.command.PostponedActionsListCommand
import ru.yandex.direct.chassis.entity.telegram.command.ReleasesForDeployCommand
import ru.yandex.direct.chassis.util.DeployException
import ru.yandex.direct.chassis.util.DeployService
import ru.yandex.direct.chassis.util.DeployStage
import ru.yandex.direct.chassis.util.DeployStatusSummary
import ru.yandex.direct.chassis.util.abc.AbcClient
import ru.yandex.direct.chassis.util.abc.ScheduleSlug
import ru.yandex.direct.chassis.util.abc.ServiceSlug
import ru.yandex.direct.chassis.util.calendar.HolidayService
import ru.yandex.direct.chassis.util.calendar.HolidayType
import ru.yandex.direct.scheduler.Hourglass
import ru.yandex.direct.scheduler.support.DirectJob
import ru.yandex.direct.staff.client.StaffClient
import ru.yandex.direct.telegram.client.TelegramClient
import ru.yandex.direct.telegram.client.api.ParseMode
import java.time.DayOfWeek
import java.time.Duration
import java.time.Instant
import java.time.LocalDate
import java.time.LocalDateTime
import java.time.LocalTime

private val RELEASES_FOR_DEPLOY_SLOTS: Map<DayOfWeek, Set<LocalTime>> = mapOf(
    DayOfWeek.MONDAY to
        setOf(LocalTime.parse("11:15"), LocalTime.parse("14:00"), LocalTime.parse("16:00")),
    DayOfWeek.TUESDAY to
        setOf(LocalTime.parse("11:00"), LocalTime.parse("14:00"), LocalTime.parse("16:00")),
    DayOfWeek.WEDNESDAY to
        setOf(LocalTime.parse("11:00"), LocalTime.parse("14:00"), LocalTime.parse("16:00")),
    DayOfWeek.THURSDAY to
        setOf(LocalTime.parse("11:00"), LocalTime.parse("14:00"), LocalTime.parse("16:00")),
    DayOfWeek.FRIDAY to
        setOf(LocalTime.parse("11:00"), LocalTime.parse("14:30")),

    // Если перенос на выходные
    DayOfWeek.SATURDAY to
        setOf(LocalTime.parse("11:00"), LocalTime.parse("14:30")),
    DayOfWeek.SUNDAY to
        setOf(LocalTime.parse("11:00"), LocalTime.parse("14:30")),
)

@Component
@Hourglass(cronExpression = "0 */15 * * * ?")
class ReleasesForDeployScheduledCommand(
    private val releasesForDeployCommand: ReleasesForDeployCommand,
    private val holidayService: HolidayService,
) : DirectJob() {
    private val logger = LoggerFactory.getLogger(ReleasesForDeployScheduledCommand::class.java)

    private val slotTolerance = Duration.ofMinutes(5)

    override fun execute() {
        val now = LocalDateTime.now()
        val today = LocalDate.now()

        val holiday = holidayService.getHoliday(today)
        if (holiday.type != HolidayType.WEEKDAY) {
            logger.info("Today is holiday, exiting: $holiday")
            return
        }

        val slots: Set<LocalTime> = RELEASES_FOR_DEPLOY_SLOTS[today.dayOfWeek].orEmpty()
        val slot: LocalTime? = slots.find { slot ->
            // Проверяем что от слота до текущего времени прошло не больше допустимой погрешности [slotTolerance]
            Duration.between(now, LocalDateTime.of(today, slot)).abs() < slotTolerance
        }

        if (slot == null) {
            logger.info("No slot found")
            return
        }

        logger.info("Found slot: $slot")
        releasesForDeployCommand.handle(TelegramChats.DIRECT_ADMIN_CHAT)
    }
}

@Component
@Hourglass(cronExpression = "0 0 11 * * MON")
class DutySwitchInfoNotification(
    private val telegram: TelegramClient,
    private val postponedActionsListCommand: PostponedActionsListCommand
) : DirectJob() {
    override fun execute() {
        var message = """
                Пора переключать дежурство по бекенду!

                Дока заступающему, переключение: https://docs.yandex-team.ru/direct-dev/duty/duty_start_shift
                Дашборд дежурства: https://st.yandex-team.ru/dashboard/67567
                Заполнить фидбек про прошедшее дежурство: https://forms.yandex-team.ru/surveys/106497/
            """.trimIndent()
        val postponedActionsMessage = postponedActionsListCommand.getListPostponedActionsTicketsMessage()
        if (postponedActionsMessage != null) {
            message += "\n\n"
            message += postponedActionsMessage
        }
        telegram.sendMessage(TelegramChats.DIRECT_ADMIN_CHAT, message, parseMode = ParseMode.HTML)
    }
}

/**
 * Уведомляет о выкладках, остановившихся на ожидании апрува локации
 *
 * Если есть активные выкладки, начатые более 45 минут назад, в которых приложение полностью выложено
 * на разрешённые локации, но выкладка ещё не завершена, то бот пишет уведомление в чат
 */
@Component
@Hourglass(cronExpression = "0 */10 * * * *")
class WaitingForApproveLocationsNotification(
    private val telegram: TelegramClient,
    private val deployService: DeployService,
    private val directReleasesService: DirectReleasesService,
    private val abcClient: AbcClient,
    private val staffClient: StaffClient,
) : DirectJob() {
    companion object {
        private val logger = LoggerFactory.getLogger(WaitingForApproveLocationsNotification::class.java)

        // регулярка для извлечения логина человека, запускавшего деплой, и ссылки на граф в CI
        // формат https://a.yandex-team.ru/arcadia/direct/web/a.yaml?rev=r9481255#L395
        private val DEPLOY_TICKET_TITLE_PATTERN =
            Pattern.compile("^([\\w\\d-]+)@ ticket: (?:[\\w\\d-]+) url: ([^ ]+) Release.*\$")
    }

    override fun execute() {
        val readyReleasesApps = directReleasesService.getReadyReleasesByApp().keys
        logger.info("Open releases: $readyReleasesApps")
        if (readyReleasesApps.isEmpty()) {
            return
        }

        for (app in readyReleasesApps) {
            val stage = DeployStage.PRODUCTION
            val stageNameList = app.yaDeployStages[stage.label] ?: continue

            var deployStatusSummaryList: List<DeployStatusSummary>
            try {
                deployStatusSummaryList = stageNameList.map {
                    deployService.getDeployStatusSummary(app, it, stage.label)
                }
            } catch (e: DeployException) {
                logger.warn("Skipping app ${app.name} with deploy error: ${e.message}")
                continue
            }
            logger.info("Got deployStatusSummaryList = $deployStatusSummaryList")

            val waitingForApproveList = deployStatusSummaryList
                .filter { !it.isDeployTicketClosed }
                .filter { it.isDeployWaitingForApproveLocation }

            val now = Instant.now()
            for (deployStatusSummary in waitingForApproveList) {
                val deploymentDuration: Duration = Duration.between(deployStatusSummary.deployTargetSpecTimestamp, now)
                logger.info("Elapsed $deploymentDuration since start of ${deployStatusSummary.app.name} deployment")

                if (deploymentDuration < Duration.ofMinutes(45)) {
                    continue
                }

                var login: String? = null
                var ciGraphUrl: String? = null
                if (deployStatusSummary.deployTicketTitle != null) {
                    val matcher = DEPLOY_TICKET_TITLE_PATTERN.matcher(deployStatusSummary.deployTicketTitle)
                    if (matcher.matches()) {
                        login = matcher.group(1)
                        ciGraphUrl = matcher.group(2)
                    }
                }
                if (login == null) {
                    login = abcClient
                        .getCurrentDutyLogins(ServiceSlug.DIRECT_APP_DUTY, ScheduleSlug.DIRECT_PRODUCTION_DUTY)
                        .firstOrNull()?.login
                }

                val telegramLogin: String? = login?.let {
                    TelegramUtils.getTelegramLogins(staffClient, listOf(it)).firstOrNull()
                }

                val appLink = if (ciGraphUrl != null) htmlLink(app.name, ciGraphUrl) else app.name
                telegram.sendMessage(
                    TelegramChats.DIRECT_ADMIN_CHAT,
                    """
                        Приложение $appLink ожидает подтверждения оставшихся локаций
                        Прошло минут с момента начала выкладки: ${deploymentDuration.toMinutes()}

                        ${if (telegramLogin != null) "@$telegramLogin" else ""}
                    """.trimIndent(),
                    ParseMode.HTML
                )
            }
        }
    }
}
