package ru.yandex.direct.chassis.entity.telegram

import org.slf4j.Logger
import org.slf4j.LoggerFactory
import ru.yandex.direct.env.Environment
import ru.yandex.direct.scheduler.Hourglass
import ru.yandex.direct.scheduler.support.DirectJob
import ru.yandex.direct.telegram.client.TelegramClient
import ru.yandex.direct.telegram.client.TelegramClientException
import ru.yandex.direct.telegram.client.api.Update
import ru.yandex.direct.utils.InterruptedRuntimeException
import java.time.Duration
import java.time.LocalDateTime

@Hourglass(periodInSeconds = 0)
class TelegramPollingBot(
    private val telegramClient: TelegramClient,
    private val telegramUpdateHandler: TelegramUpdateHandler,
) : DirectJob() {

    companion object {
        private val logger: Logger = LoggerFactory.getLogger(TelegramPollingBot::class.java)

        private val iterationTime = if (Environment.get().isDevelopment) {
            Duration.ofDays(1)
        } else {
            Duration.ofMinutes(5)
        }

        private const val updateLimit: Int = 100
        private const val timeoutSeconds: Int = 5
    }

    // https://core.telegram.org/bots/api#getupdates
    override fun execute() {
        val borderTime = LocalDateTime.now() + iterationTime

        var offset = 0
        while (LocalDateTime.now() < borderTime) {
            val updates = try {
                telegramClient.getUpdates(
                    offset = offset,
                    limit = updateLimit,
                    timeout = timeoutSeconds,
                )
            } catch (e: TelegramClientException) {
                logger.error("Failed to fetch updates, offset = $offset", e)
                return
            }

            handleUpdates(updates)

            if (updates.isNotEmpty()) {
                offset = updates.maxOf { it.updateId } + 1
            }
        }

        // confirm handled updates
        telegramClient.getUpdates(offset = offset, limit = 1)
    }

    private fun handleUpdates(updates: List<Update>) {
        updates.forEach { update ->
            try {
                telegramUpdateHandler.handle(update)
            } catch (e: InterruptedException) {
                Thread.currentThread().interrupt()
                throw InterruptedRuntimeException(e)
            } catch (e: Exception) {
                logger.error("Failed to handle update $update, dropping it", e)
            }
        }
    }

}
