package ru.yandex.direct.chassis.entity.telegram

import org.slf4j.Logger
import org.slf4j.LoggerFactory
import org.springframework.stereotype.Component
import ru.yandex.direct.chassis.entity.telegram.command.CommandHandler
import ru.yandex.direct.chassis.entity.telegram.handler.MessageHandler
import ru.yandex.direct.chassis.entity.telegram.model.AllowedChats
import ru.yandex.direct.chassis.util.Utils
import ru.yandex.direct.chassis.util.staff.StaffTelegramAccountsProvider
import ru.yandex.direct.telegram.client.TelegramClient
import ru.yandex.direct.telegram.client.api.Message
import ru.yandex.direct.telegram.client.api.ParseMode
import ru.yandex.direct.telegram.client.api.Update
import ru.yandex.direct.telegram.client.api.User

@Component
class TelegramUpdateHandler(
    private val telegramClient: TelegramClient,
    private val messageHandlers: List<MessageHandler>,
    private val commandHandlers: List<CommandHandler>,
    private val staffTelegramAccountsProvider: StaffTelegramAccountsProvider,
) {
    private val logger: Logger = LoggerFactory.getLogger(TelegramUpdateHandler::class.java)

    private val myself: User by lazy {
        telegramClient.getMe()
    }

    fun handle(update: Update) {
        logger.info("Handling update: $update")

        if (update.message != null) {
            val message = update.message!!

            val command = message.text
                ?.let { TelegramUtils.parseCommand(it) }
            if (command != null && command.checkTarget(myself.username)) {
                val handler = commandHandlers
                    .firstOrNull { it.command == command.text }
                if (handler != null) {
                    if (!checkMessageIsAllowed(message, handler.allowedChats)) {
                        logger.info("Received disallowed message from ${message.from}, chat ${message.chat}, skipping")
                        return
                    }
                    wrapMessageHandler(message) {
                        handler.handle(message, command)
                    }
                    return
                }
            }

            val handler = messageHandlers
                .firstOrNull { it.matches(message) }
            if (handler != null) {
                if (!checkMessageIsAllowed(message, handler.allowedChats)) {
                    logger.info("Received non-allowed message from ${message.from}, chat ${message.chat}, skipping")
                    return
                }
                wrapMessageHandler(message) {
                    handler.handle(message)
                }
                return
            }
        }
    }

    private fun checkMessageIsAllowed(message: Message, allowedChats: AllowedChats): Boolean {
        return allowedChats.isAllowedUnconditionally(message.chat)
            || allowedChats.isAllowedWithUserCheck(message.chat)
            && message.from?.username in staffTelegramAccountsProvider.getLogins()
    }

    private fun wrapMessageHandler(message: Message, block: () -> Unit) {
        try {
            block()
        } catch (e: Exception) {
            val logLink = TelegramUtils.htmlLink("лог", Utils.currentLogViewerLink())
            telegramClient.sendMessage(
                message.chat.id,
                "Не получилось, $logLink ☹️",
                parseMode = ParseMode.HTML,
            )
            throw e
        }
    }
}
