package ru.yandex.direct.chassis.entity.telegram.command

import org.slf4j.LoggerFactory
import org.springframework.stereotype.Component
import ru.yandex.direct.chassis.entity.telegram.TelegramChats
import ru.yandex.direct.chassis.entity.telegram.TelegramUtils
import ru.yandex.direct.chassis.entity.telegram.model.AllowedChats
import ru.yandex.direct.chassis.entity.telegram.model.Command
import ru.yandex.direct.chassis.util.abc.AbcClient
import ru.yandex.direct.chassis.util.abc.ScheduleSlug
import ru.yandex.direct.chassis.util.abc.ServiceSlug
import ru.yandex.direct.staff.client.StaffClient
import ru.yandex.direct.telegram.client.TelegramClient
import ru.yandex.direct.telegram.client.api.Message

@Component
class DutyCallCommand(
    private val telegram: TelegramClient,
    private val abcClient: AbcClient,
    private val staffClient: StaffClient,
) : CommandHandler {
    private val logger = LoggerFactory.getLogger(DutyCallCommand::class.java)
    private val dutyCallLimit: Int = 5

    override val command = "duty_call"

    override val allowedChats = AllowedChats.fromChatIds(
        TelegramChats.DIRECT_PROD_PROBLEMS,
        TelegramChats.DIRECT_APP_DUTY,
    )

    private fun getDutyLogins(): Set<String> {
        return abcClient.getCurrentDutyLogins(ServiceSlug.DIRECT_APP_DUTY, ScheduleSlug.DIRECT_PRODUCTION_DUTY)
            .map { it.login }
            .toSet()
    }

    override fun handle(update: Message, command: Command) {
        val author: String? = update.from?.username

        val staffDutyLogins = getDutyLogins()
        val telegramDutyLogins = TelegramUtils.getTelegramLogins(staffClient, staffDutyLogins)
        if (telegramDutyLogins.size > dutyCallLimit) {
            logger.warn("Too many telegram logins: $telegramDutyLogins")
        }

        val loginsString = telegramDutyLogins.take(dutyCallLimit).joinToString { "@$it" }
        telegram.sendMessage(
            update.chat.id,
            text = """
                @$author говорит, что в проде, вероятно, массовая критичная проблема.

                $loginsString - посмотрите, пожалуйста.
            """.trimIndent(),
        )
    }
}

@Component
class DutyRespondCommand(private val telegram: TelegramClient) : CommandHandler {
    override val command = "duty_respond"

    override val allowedChats = AllowedChats.fromChatIds(
        TelegramChats.DIRECT_PROD_PROBLEMS,
        TelegramChats.DIRECT_APP_DUTY,
    )

    override fun handle(update: Message, command: Command) {
        val author: String? = update.from?.username
        telegram.sendMessage(
            update.chat.id,
            text = """
                @$author говорит, что занимается проблемой.

                @$author:
                  * Оцени масштаб аварии: https://nda.ya.ru/3VrXnL
                  * Затрудняешься - сразу проси помощи (https://nda.ya.ru/3VrXza)
                  * Общая инструкция: https://nda.ya.ru/3VrXmq
            """.trimIndent(),
        )
    }
}

@Component
class DutyCancelCommand(private val telegram: TelegramClient) : CommandHandler {
    override val command = "duty_cancel"

    override val allowedChats = AllowedChats.fromChatIds(
        TelegramChats.DIRECT_PROD_PROBLEMS,
        TelegramChats.DIRECT_APP_DUTY,
    )

    override fun handle(update: Message, command: Command) {
        val author: String? = update.from?.username
        telegram.sendMessage(
            update.chat.id,
            text = """
                @$author говорит, что критичных проблем в проде нет, действий от дежурных не требуется.
            """.trimIndent(),
        )
    }
}
