package ru.yandex.direct.chassis.entity.telegram.command

import org.slf4j.LoggerFactory
import org.springframework.stereotype.Component
import ru.yandex.direct.chassis.entity.release.service.ReleaseChecker
import ru.yandex.direct.chassis.entity.release.service.ReleaseCheckerFactory
import ru.yandex.direct.chassis.entity.startrek.ReleaseDependencyService
import ru.yandex.direct.chassis.entity.telegram.ReleasesForDeployScheduledCommand
import ru.yandex.direct.chassis.entity.telegram.TelegramChats
import ru.yandex.direct.chassis.entity.telegram.TelegramUtils.htmlLink
import ru.yandex.direct.chassis.entity.telegram.model.AllowedChats
import ru.yandex.direct.chassis.entity.telegram.model.Command
import ru.yandex.direct.chassis.util.startrek.StartrekHelper
import ru.yandex.direct.chassis.util.startrek.StartrekIssueTypeKey
import ru.yandex.direct.chassis.util.startrek.StartrekQueue
import ru.yandex.direct.chassis.util.startrek.StartrekStatusKey
import ru.yandex.direct.hourglass.client.HourglassClient
import ru.yandex.direct.hourglass.storage.Job
import ru.yandex.direct.telegram.client.TelegramClient
import ru.yandex.direct.telegram.client.api.ChatAction
import ru.yandex.direct.telegram.client.api.Message
import ru.yandex.direct.telegram.client.api.ParseMode
import ru.yandex.startrek.client.model.Issue

@Component
class ReleasesForDeployCommand(
    private val startrek: StartrekHelper,
    private val telegram: TelegramClient,
    private val releaseDependencyService: ReleaseDependencyService,
    private val releaseCheckerFactory: ReleaseCheckerFactory,
    private val hourglassClient: HourglassClient,
) : CommandHandler {

    companion object {
        private val logger = LoggerFactory.getLogger(ReleasesForDeployCommand::class.java)

        private const val NEW_CI_INSTRUCTION =
            "https://docs.yandex-team.ru/direct-dev/processes/regulations_deploy_releases_newci"
        private const val OTHER_RELEASES_INSTRUCTION = "https://wiki.yandex-team.ru/jeri/app-duty/releases/"
    }

    override val command: String = "releases_for_deploy"

    override val allowedChats = AllowedChats.fromChatIds(
        TelegramChats.DIRECT_ADMIN_CHAT,
    )

    override fun handle(update: Message, command: Command) {
        when (command.args.firstOrNull()) {
            "pause" -> pauseScheduled(update)
            "resume" -> resumeScheduled(update)
            else -> {
                telegram.sendChatAction(update.chat.id, ChatAction.TYPING)
                handle(update.chat.id)
            }
        }
    }

    private fun scheduledTasks(): List<Job> {
        val tasks = hourglassClient.tasks
            .filter { it.taskId().name() == ReleasesForDeployScheduledCommand::class.qualifiedName }

        logger.info("Releases for deploy tasks: $tasks")
        return tasks
    }

    private fun pauseScheduled(update: Message) {
        if (update.chat.id != TelegramChats.DIRECT_ADMIN_CHAT) {
            telegram.sendMessage(
                update.chat.id,
                """Управлять уведомлениями о выкладке можно только в админском чате!"""
            )
            return
        }

        scheduledTasks().forEach { job ->
            hourglassClient.pauseTask(job)
        }

        telegram.sendMessage(
            update.chat.id,
            """
                ⏸ Уведомления о релизах для выкладки приостановлены.
                Чтобы возобновить: <code>/$command resume</code>
            """.trimIndent(),
            parseMode = ParseMode.HTML,
        )
    }

    private fun resumeScheduled(update: Message) {
        if (update.chat.id != TelegramChats.DIRECT_ADMIN_CHAT) {
            telegram.sendMessage(
                update.chat.id,
                """Управлять уведомлениями о выкладке можно только в админском чате!"""
            )
            return
        }

        scheduledTasks().forEach { job ->
            hourglassClient.continueTask(job)
        }

        telegram.sendMessage(
            update.chat.id,
            """
                ▶️ Уведомления о релизах для выкладки возобновлены.
                Чтобы приостановить: <code>/$command pause</code>
            """.trimIndent(),
            parseMode = ParseMode.HTML,
        )
    }

    fun handle(chatId: Long) {
        val releases: List<Issue> = startrek.findIssues(
            """
                Queue: ${StartrekQueue.DIRECT}
                Type: ${StartrekIssueTypeKey.RELEASE}
                Status: ${StartrekStatusKey.READY_TO_DEPLOY}, ${StartrekStatusKey.RM_ACCEPTANCE}
                "Sort by": key asc
            """.trimIndent()
        )

        val dependencyGraph = releaseDependencyService.buildDependencyGraph()
        val unclosedReleasesByApp = releaseCheckerFactory.getUnclosedReleasesByApp()

        val releasesToNotify: List<ReleaseChecker> = releases
            .mapNotNull { releaseCheckerFactory.createReleaseContext(it, dependencyGraph, unclosedReleasesByApp) }
            .filter { it.shouldNotify() }

        if (releasesToNotify.isEmpty()) {
            telegram.sendMessage(chatId, text = "Нет релизов для выкладки")
            return
        }

        var message = "Ждут выкладки релизы:\n"

        val (newCiReleases, otherReleases) = releasesToNotify
            .partition { it.isNewCiApp() }

        if (newCiReleases.isNotEmpty()) {
            message += "\n<b>NewCI приложения, ${htmlLink("регламент выкладки", NEW_CI_INSTRUCTION)}:</b>\n"
            message += newCiReleases.joinToString("\n") { it.buildNotification() }
            message += "\n"
        }

        if (otherReleases.isNotEmpty()) {
            message += "\n<b>Другие приложения, ${htmlLink("инструкция", OTHER_RELEASES_INSTRUCTION)}:</b>\n"
            message += otherReleases.joinToString("\n") { it.buildNotification() }
            message += "\n"
        }

        telegram.sendMessage(chatId, text = message, parseMode = ParseMode.HTML)
    }
}
