package ru.yandex.direct.chassis.entity.telegram.handler

import org.springframework.stereotype.Component
import ru.yandex.direct.chassis.entity.release.model.ReleaseError
import ru.yandex.direct.chassis.entity.startrek.DependencyGraph
import ru.yandex.direct.chassis.entity.startrek.ReleaseDependencyService
import ru.yandex.direct.chassis.entity.telegram.TelegramChats
import ru.yandex.direct.chassis.entity.telegram.TelegramUtils
import ru.yandex.direct.chassis.entity.telegram.model.AllowedChats
import ru.yandex.direct.chassis.entity.release.service.ReleaseCheckerFactory
import ru.yandex.direct.chassis.util.nonCf
import ru.yandex.direct.chassis.util.startrek.StartrekComponent
import ru.yandex.direct.chassis.util.startrek.StartrekHelper
import ru.yandex.direct.chassis.util.startrek.StartrekIssueTypeKey
import ru.yandex.direct.chassis.util.startrek.StartrekQueue
import ru.yandex.direct.chassis.util.startrek.StartrekStatusKey
import ru.yandex.direct.telegram.client.TelegramClient
import ru.yandex.direct.telegram.client.api.ChatAction
import ru.yandex.direct.telegram.client.api.Message
import ru.yandex.direct.telegram.client.api.ParseMode
import ru.yandex.startrek.client.model.Issue

@Component
class CheckDeployHandler(
    private val startrek: StartrekHelper,
    private val telegram: TelegramClient,
    private val releaseDependencyService: ReleaseDependencyService,
    private val releaseCheckerFactory: ReleaseCheckerFactory,
) : MessageHandler {

    companion object {
        private val DEPLOY_MESSAGE_REGEX: Regex = ("вылож(?:ить|ите|ишь|им|у)" +
            "|обнов(?:ить|ите|ишь|им|лю)" +
            "|выкат(?:ить|ите|ишь|им|ки|ке)|выкачу" +
            "|катн(?:ите|и|у|ешь)" +
            "|limtest[12]?|limtest" +
            "|залоч(?:у|ите|ишь)").toRegex(RegexOption.IGNORE_CASE)

        private val TICKET_KEY_REGEX: Regex =
            """(${StartrekQueue.DIRECT}|${StartrekQueue.DIRECTMIGR})-\d+"""
                .toRegex(RegexOption.IGNORE_CASE)

        private val LIMTEST_APPROVERS: List<String> =
            listOf("raketa", "sudar", "ppalex", "gerdler", "hmepas", "maxlog", "pavryabov", "buhter")

        private val LIMTEST_APPROVE_REGEX: Regex =
            """(?:можно\s*)?(?:выкладывать|выкатывать|катить|выкладывай|выкладывайте|выложите|подтверждаю)"""
                .toRegex(RegexOption.IGNORE_CASE)
    }

    override val allowedChats = AllowedChats.fromChatIds(
        TelegramChats.DIRECT_ADMIN_CHAT,
    )

    override fun matches(update: Message): Boolean {
        val text = update.text ?: return false
        return text.contains(DEPLOY_MESSAGE_REGEX)
    }

    override fun handle(update: Message) {
        val issueKeys = TICKET_KEY_REGEX.findAll(update.text!!)
            .map { it.value }
            .toList()

        val issues = startrek.findIssuesByKeys(issueKeys)
        if (issues.isEmpty()) {
            return
        }

        telegram.sendChatAction(update.chat.id, ChatAction.TYPING)

        val dependencyGraph = releaseDependencyService.buildDependencyGraph()
        val unclosedReleasesByApp = releaseCheckerFactory.getUnclosedReleasesByApp()

        var message = "Информация по выкладке:\n"

        message += issues.joinToString("\n") { issue ->
            val errors = getErrors(issue, dependencyGraph, unclosedReleasesByApp)
            if (errors.isNotEmpty()) {
                "${TelegramUtils.issueLink(issue)}:\n" +
                    errors.joinToString("\n")
            } else {
                "${TelegramUtils.issueLink(issue)}: ок"
            }
        }

        telegram.sendMessage(update.chat.id, message, parseMode = ParseMode.HTML)
    }

    private fun getErrors(
        issue: Issue,
        dependencyGraph: DependencyGraph,
        unclosedReleasesByApp: Map<String, List<Issue>>,
    ): List<ReleaseError> {
        return when {
            issue.type.key == StartrekIssueTypeKey.RELEASE -> {
                releaseCheckerFactory.createReleaseContext(issue, dependencyGraph, unclosedReleasesByApp)
                    ?.collectErrors()
                    ?: listOf(ReleaseError.warn("неизвестное приложение"))
            }

            issue.queue.key == StartrekQueue.DIRECTMIGR -> {
                if (issue.status.key == StartrekStatusKey.OPEN
                    || issue.status.key == StartrekStatusKey.NEED_INFO
                ) {
                    listOf(ReleaseError.error("неправильный статус (${issue.status.display})"))
                } else {
                    listOf()
                }
            }

            StartrekComponent.LIMTEST in issue.components.nonCf().map { it.display }
                || issue.summary.contains("выложить на среду ограниченного тестирования") -> {
                checkLimtest(issue)
            }

            else -> listOf(ReleaseError.warn("неизвестный тип тикета, действуйте на собственное усмотрение"))
        }
    }

    private fun checkLimtest(issue: Issue): List<ReleaseError> {
        val errors: MutableList<ReleaseError> = mutableListOf()
        checkLimtestApprove(issue, errors)
        return errors
    }

    private fun checkLimtestApprove(issue: Issue, errors: MutableList<ReleaseError>) {
        val hasApprove = issue.comments.toList().nonCf()
            .filter { it.createdBy.login in LIMTEST_APPROVERS }
            .mapNotNull { it.text.orElse(null as String?) }
            .any { it.contains(LIMTEST_APPROVE_REGEX) }

        if (!hasApprove) {
            errors += ReleaseError.warn("нет комментария-подтверждения от ответственных")
        }
    }
}
