package ru.yandex.direct.chassis.entity.telegram.scheduled

import org.springframework.stereotype.Component
import ru.yandex.direct.chassis.entity.telegram.TelegramChats
import ru.yandex.direct.chassis.entity.telegram.TelegramUtils
import ru.yandex.direct.chassis.entity.telegram.TelegramUtils.htmlLink
import ru.yandex.direct.chassis.entity.telegram.model.OwnersDiffTableRow
import ru.yandex.direct.scheduler.Hourglass
import ru.yandex.direct.scheduler.support.DirectJob
import ru.yandex.direct.staff.client.StaffClient
import ru.yandex.direct.telegram.client.TelegramClient
import ru.yandex.direct.telegram.client.api.ParseMode
import ru.yandex.direct.ytwrapper.client.YtProvider
import ru.yandex.direct.ytwrapper.model.YtCluster
import ru.yandex.direct.ytwrapper.model.YtTable


private const val FEAUTURES_COUNT_DIFF_TABLE_PATH = "//home/direct/test/features/owners_tables/owners_diff"
private const val CAN_BE_DELETED_FROM_CODE_FEATURES_LINK =
    "https://datalens.yandex-team.ru/4sydz8jacps0w-statistika-fichey?can_be_deleted_from_code=true"
private const val SHOULD_BE_FULLY_ENABLED_FEATURES_LINK =
    "https://datalens.yandex-team.ru/4sydz8jacps0w-statistika-fichey?outdated=true&fully_enabled=false"

/**
 * Уведомляет ответственных, которые увеличили число фич в базе
 * за последние две недели
 */
@Component
@Hourglass(cronExpression = "0 0 11 * * MON")
class FeaturesOwnersNotificationMessage(
    private val telegram: TelegramClient, private val ytProvider: YtProvider, private val staffClient: StaffClient
) : DirectJob() {
    override fun execute() {
        var message = "<b>Статистика по увеличению числа просроченных фич за последние две недели</b>\n"

        var canBeDeletedFromCode = 0L
        var shouldBeEnabled = 0L
        val featuresCountByStaffLogin: MutableMap<String, Pair<Long, Long>> = mutableMapOf()

        val ytOperator = ytProvider.getOperator(YtCluster.HAHN)
        ytOperator.readTable(YtTable(FEAUTURES_COUNT_DIFF_TABLE_PATH), OwnersDiffTableRow()) { row ->
            canBeDeletedFromCode += row.canBeDeletedFromCode
            shouldBeEnabled += row.shouldBeEnabled

            if (row.outdatedNow - row.outdatedTwoWeeksAgo > 0) {
                featuresCountByStaffLogin[row.originalOwner] = Pair(row.outdatedNow, row.outdatedTwoWeeksAgo)
            }
        }

        message = addStaffUsersInformationToMessage(featuresCountByStaffLogin, message)

        message = addLinksToMessage(message, canBeDeletedFromCode, shouldBeEnabled)

        telegram.sendMessage(
            TelegramChats.FEATURES_DIFF_NOTIFICATION_CHAT,
            text = message.trimIndent(),
            parseMode = ParseMode.HTML
        )
    }

    private fun addStaffUsersInformationToMessage(
        featuresCountByStaffLogin: Map<String, Pair<Long, Long>>,
        message: String
    ): String {
        var updatedMessage = message
        if (featuresCountByStaffLogin.isNotEmpty()) {
            val telegramLoginsByStaffLogin =
                TelegramUtils.getTelegramLoginsByStaffLogin(staffClient, featuresCountByStaffLogin.keys)

            featuresCountByStaffLogin.entries
                .sortedBy {(_, featuresCounts) -> featuresCounts.second - featuresCounts.first}
                .forEach { (staffLogin, featuresCounts) ->
                    updatedMessage += if (staffLogin == "no owner") {
                        "Без ответственного,  "
                    } else {
                        val telegramLogin = telegramLoginsByStaffLogin[staffLogin]
                        (if (telegramLogin != null) "tg: $telegramLogin, " else "") +
                            "staff: ${
                                htmlLink(
                                    staffLogin,
                                    "https://staff.yandex-team.ru/$staffLogin"
                                )
                            }, "
                    }

                    updatedMessage += htmlLink(
                        "${featuresCounts.second} -> ${featuresCounts.first}, diff: +${featuresCounts.first - featuresCounts.second}\n",
                        "https://datalens.yandex-team.ru/4sydz8jacps0w-statistika-fichey?owner_name=$staffLogin&outdated=true"
                    )
                }
        }
        return updatedMessage
    }

    private fun addLinksToMessage(
        message: String,
        canBeDeletedFromCode: Long,
        shouldBeEnabled: Long
    ): String {
        var updatedMessage = message
        val canBeDeletedFromCodeLink = htmlLink(
            "Число фич, доступных для удаления из кода: $canBeDeletedFromCode",
            CAN_BE_DELETED_FROM_CODE_FEATURES_LINK
        )
        val shouldBeFullyEnabledLink = htmlLink(
            "Число фич, которые необходимо включить на 100 процентов или выключить и удалить: $shouldBeEnabled",
            SHOULD_BE_FULLY_ENABLED_FEATURES_LINK
        )

        updatedMessage += """

                $canBeDeletedFromCodeLink
                (Включенные на 100 процентов и без явного выключения на конкретных клиентах)

                $shouldBeFullyEnabledLink
                (Обновлявшиеся больше 180 дней назад и неполностью включенные)
                """.trimIndent()

        return updatedMessage
    }
}
