package ru.yandex.direct.chassis.monitor

import org.slf4j.LoggerFactory
import org.springframework.beans.factory.annotation.Autowired
import org.tmatesoft.svn.core.SVNURL
import org.tmatesoft.svn.core.wc.SVNClientManager
import org.tmatesoft.svn.core.wc.SVNRevision
import ru.yandex.direct.chassis.util.DirectAppsConfProvider
import ru.yandex.direct.chassis.model.Commit
import ru.yandex.direct.chassis.model.CommitDelay
import ru.yandex.direct.chassis.repository.CommitDeployRepository
import ru.yandex.direct.chassis.repository.CommitRepository
import ru.yandex.direct.chassis.repository.DeployRepository
import ru.yandex.direct.scheduler.Hourglass
import ru.yandex.direct.scheduler.support.DirectJob
import ru.yandex.direct.utils.DateTimeUtils
import java.time.Duration


private val logger = LoggerFactory.getLogger(CommitMonitor::class.java)

// Коммит около 1 января 2020
private const val START_COMMIT = 6190000L

private val DIRECT_TRUNK_URL: SVNURL = SVNURL.parseURIEncoded("svn+ssh://arcadia.yandex.ru/arc/trunk/arcadia/direct")

private val VERSION_REGEX = """1\.(\d+)-1""".toRegex()
private val BRANCH_VERSION_REGEX = """1\.(\d+)\.(\d+)-1""".toRegex()

private val HOTFIX_MERGE_PATTERN = """RELEASE: Merged ([\d\s,]+)""".toRegex()

private const val JAVA_BRANCH_URL_PATTERN = "svn+ssh://arcadia.yandex.ru/arc/branches/direct/release/%s/%d"
private const val PERL_BRANCH_URL_PATTERN = "svn+ssh://arcadia.yandex.ru/arc/branches/direct/release/perl/release-%d"

class DeployVersion(val baseRevision: Long, val branchRevision: Long = baseRevision) {
    val isBranch = baseRevision != branchRevision
}

@Hourglass(periodInSeconds = 60)
class CommitMonitor @Autowired constructor(
    private val commitRepository: CommitRepository,
    private val deployRepository: DeployRepository,
    private val commitDelayRepository: CommitDeployRepository,
    private val svnClientManager: SVNClientManager,
    private val directAppsConfProvider: DirectAppsConfProvider,
) : DirectJob() {

    override fun execute() {
        fetchNewCommits()
        processNewDeploys()
    }

    private fun fetchNewCommits() {
        val lastCommit = commitRepository.getLastCommit()
        val startRevision = lastCommit?.revision ?: START_COMMIT

        val commits = mutableListOf<Commit>()

        logger.info("Getting commits after revision {}", startRevision)

        val svnRevision = SVNRevision.create(startRevision + 1)
        svnClientManager.logClient.doLog(
            DIRECT_TRUNK_URL, null,
            svnRevision, svnRevision, SVNRevision.HEAD,
            false, false, 10000
        ) {
            commits.add(
                Commit()
                    .withRevision(it.revision)
                    .withTime(DateTimeUtils.instantToMoscowDateTime(it.date.toInstant()))
            )
        }

        logger.info("Got {} commits", commits.size)

        val commitDelayRecords = commits.flatMap { commit ->
            directAppsConfProvider.getDirectAppsConf().map { app ->
                CommitDelay()
                    .withApp(app.name)
                    .withCommit(commit)
                    .withDeployDelay(Duration.ofSeconds(-1))
                    .withDeployDelayWoWeekend(Duration.ofSeconds(-1))
                    .withDeployDelayWoWeekendAndNight(Duration.ofSeconds(-1))
            }
        }

        commitRepository.addCommits(commits)
        commitDelayRepository.addDelays(commitDelayRecords)
    }

    private fun processNewDeploys() {
        for (app in directAppsConfProvider.getDirectAppsConf()) {
            val deploys = deployRepository.getDeploysForApp(app.name).map { it to parseVersion(it.version) }
                .sortedWith(
                    compareBy(
                        { it.second.baseRevision },
                        { it.second.branchRevision },
                        { it.first.deployTime })
                )

            for ((index, deployWithVersion) in deploys.withIndex()) {
                val (deploy, version) = deployWithVersion

                if (!deploy.delayCounted) {
                    logger.info("Counting delay for deploy {}", deploy)
                    val previousDeployedVersion = if (index > 0) deploys[index - 1].second else DeployVersion(0)

                    val fromExclusive = previousDeployedVersion.baseRevision
                    val toInclusive = version.baseRevision
                    val excluding = getHotfixCommits(app.name, previousDeployedVersion)
                    val including = getHotfixCommits(app.name, version)

                    commitDelayRepository.assignCorrectedDeploy(
                        deploy,
                        fromExclusive,
                        toInclusive,
                        excluding,
                        including
                    )
                    deployRepository.setDelayCounted(deploy)
                }
            }
        }
    }

    private fun getHotfixCommits(app: String, version: DeployVersion): List<Long> {
        if (!version.isBranch) return listOf()

        val svnUrl = if (app == "direct") {
            SVNURL.parseURIEncoded(PERL_BRANCH_URL_PATTERN.format(version.baseRevision))
        } else {
            SVNURL.parseURIEncoded(JAVA_BRANCH_URL_PATTERN.format(app, version.baseRevision))
        }

        val result = mutableListOf<Long>()
        svnClientManager.logClient.doLog(
            svnUrl, null, SVNRevision.HEAD,
            SVNRevision.create(version.baseRevision), SVNRevision.create(version.branchRevision),
            false, false, 10000
        ) { logEntry ->
            val firstLine = logEntry.message.lines().first()

            val match = HOTFIX_MERGE_PATTERN.matchEntire(firstLine)
            if (match != null) {
                val (_, revisions) = match.groupValues
                revisions.split(' ', ',').mapNotNull {
                    it.toLongOrNull()
                }.forEach { revision ->
                    result.add(revision)
                }
            }
        }

        return result
    }

    private fun parseVersion(version: String): DeployVersion {
        BRANCH_VERSION_REGEX.matchEntire(version)?.groupValues?.let { (_, base, branch) ->
            return DeployVersion(base.toLong(), branch.toLong())
        }
        VERSION_REGEX.matchEntire(version)?.groupValues?.let { (_, base) ->
            return DeployVersion(base.toLong())
        }
        throw IllegalArgumentException("Invalid deploy version $version")
    }

}
