package ru.yandex.direct.chassis.monitor

import org.slf4j.LoggerFactory
import org.springframework.beans.factory.annotation.Autowired
import ru.yandex.direct.chassis.util.DirectAppsConfProvider
import ru.yandex.direct.chassis.model.Deploy
import ru.yandex.direct.chassis.repository.DeployRepository
import ru.yandex.direct.scheduler.Hourglass
import ru.yandex.direct.scheduler.support.DirectJob
import ru.yandex.direct.utils.DateTimeUtils
import ru.yandex.startrek.client.Session
import ru.yandex.startrek.client.model.Comment
import ru.yandex.startrek.client.model.Event
import ru.yandex.startrek.client.model.Issue
import java.time.Duration
import java.time.LocalDate
import java.time.LocalDateTime
import java.time.format.DateTimeFormatter

private val logger = LoggerFactory.getLogger(DeployMonitor::class.java)

private val DEPLOY_TIME_TRUNCATE_PERIOD = Duration.ofMinutes(5)
val START_FETCH_TIME_FOR_EMPTY_DB = LocalDate.parse("2020-01-01").atStartOfDay()

val COMMENT_REGEX = "Пакет версии (.+) установлен на группе продакшеновых машин.*"
    .toRegex(RegexOption.DOT_MATCHES_ALL)

@Hourglass(periodInSeconds = 60)
class DeployMonitor @Autowired constructor(
    private val deployRepository: DeployRepository,
    private val startrekSession: Session,
    private val directAppsConfProvider: DirectAppsConfProvider,
) : DirectJob() {

    private val appByTrackerComponent: Map<String, String>
        get() = directAppsConfProvider.getDirectAppsConf()
            .associate { it.trackerComponent to it.name }

    override fun execute() {
        val lastDeploy = deployRepository.getLastByDeployTime()

        val startTime = if (lastDeploy != null) {
            lastDeploy.deployTime - DEPLOY_TIME_TRUNCATE_PERIOD
        } else {
            START_FETCH_TIME_FOR_EMPTY_DB
        }

        logger.info("Fetching new deploys after {}", startTime)

        val deploys = getDeploysAfter(startTime)
        logger.info("Found {} deploys after {}", deploys.size, startTime)

        deployRepository.addDeploys(deploys)
    }

    private fun getDeploysAfter(startTime: LocalDateTime): List<Deploy> {
        val timeFormatted = DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm:ss").format(startTime)

        val query = """queue:DIRECT type:release Components: "Releases: Direct", "Releases: JavaDirect"
                |Updated: >"$timeFormatted" "Sort by": Created ASC""".trimMargin()

        val issues = startrekSession.issues().find(query).toList()
        logger.info("Found {} release tickets after {} ({}); query: '{}'", issues.size, startTime, timeFormatted, query)

        return issues.flatMap { issue ->
            getDeployComments(issue).filter { deploy ->
                deploy.deployTime > startTime
            }
        }
    }

    private fun getDeployComments(issue: Issue): List<Deploy> {
        val appComponent = issue.components
            .map { it.display }
            .firstOrNull { it in appByTrackerComponent }
        appComponent ?: return emptyList()

        val app = appByTrackerComponent[appComponent]

        val comments: List<Comment> = issue.comments.toList()
        val deploys = comments
            .filter { it.updatedBy.login == "ppc" }
            .filter { it.text.isPresent }
            .mapNotNull { comment ->
                COMMENT_REGEX.matchEntire(comment.text.get())?.let { match ->
                    Deploy().also {
                        val deployStatuses = getReleaseStates(issue)
                        it.app = app
                        it.version = match.groupValues[1]
                        it.ticketKey = issue.key
                        it.deployTime = DateTimeUtils.fromEpochMillis(comment.createdAt.millis)
                        it.delayCounted = false
                        it.rftTime = deployStatuses.getOrDefault("readyForTest", listOf())
                        it.rmAccTime = deployStatuses.getOrDefault("rmAcceptance", listOf())
                        it.rtdTime = deployStatuses.getOrDefault("readyToDeploy", listOf())
                    }
                }
            }

        fun isDuplicate(d1: Deploy, d2: Deploy) =
            Duration.between(d1.deployTime, d2.deployTime).abs() < Duration.ofMinutes(1)

        val deduplicated = mutableListOf<Deploy>()
        for (deploy in deploys) {
            if (deduplicated.isNotEmpty() && isDuplicate(deduplicated.last(), deploy)) continue

            deduplicated.add(deploy)
        }

        return deduplicated
    }

    private fun getReleaseStates(issue: Issue): Map<String, List<LocalDateTime>> {
        val updateStatusEventsWithDates = mutableMapOf<String, List<LocalDateTime>>()
        val updateStatusEvents =
            startrekSession.events().getAll(issue.key).toList().filter { it.type == "IssueWorkflow" }
                .filter { filterEvents(it, "status") }
        updateStatusEvents.forEach { event ->
            for (eventUpdateFields in event.fields) {
                if (eventUpdateFields.`field`.id != "status") continue
                for (change in eventUpdateFields.to) {
                    if (change is LinkedHashMap<*, *> && change.containsKey("key")) {
                        val up = DateTimeUtils.fromEpochMillis(event.updatedAt.millis).withNano(0)
                        updateStatusEventsWithDates[change["key"] as String] =
                            updateStatusEventsWithDates.getOrDefault(change["key"], listOf()).plus(listOf(up))
                    }
                }
            }
        }
        return updateStatusEventsWithDates.toMap()
    }

    private fun filterEvents(event: Event?, type: String): Boolean {
        return event?.fields?.any { it.field.id == type }!!
    }
}
