package ru.yandex.direct.chassis.monitor

import org.slf4j.LoggerFactory
import org.springframework.beans.factory.annotation.Autowired
import ru.yandex.direct.chassis.util.DirectAppsConfProvider
import ru.yandex.direct.chassis.model.Hotfix
import ru.yandex.direct.chassis.repository.DeployHotfixRepository
import ru.yandex.direct.scheduler.Hourglass
import ru.yandex.direct.scheduler.support.DirectJob
import ru.yandex.startrek.client.Session
import ru.yandex.startrek.client.model.Comment
import ru.yandex.startrek.client.model.Event
import ru.yandex.startrek.client.model.Issue
import java.time.LocalDateTime
import java.time.ZoneId
import java.time.format.DateTimeFormatter

private val logger = LoggerFactory.getLogger(DeployMonitor::class.java)

private val COMMENT_REGEX_HOTFIX = "В релиз включен новый код, версия: 1.* --> 1[.][0-9]+[.][0-9]+-1.*"
    .toRegex(RegexOption.DOT_MATCHES_ALL)
private val COMMENT_REGEX_TICKETS = "(DIRECT-\\d+)"
    .toRegex(RegexOption.DOT_MATCHES_ALL)
private val COMMENT_REGEX_REVISIONS = "r(\\d+)"
    .toRegex(RegexOption.DOT_MATCHES_ALL)
private val COMMENT_REGEX_DNA_REVISIONS = "0\\.[0-9]+\\.0..-(.{11})"
    .toRegex(RegexOption.DOT_MATCHES_ALL)
private val COMMENT_REGEX_HOTFIX_REVISIONS_AND_TICKETS = "r(\\d+).*\\n(DIRECT-\\d+)"
    .toRegex(RegexOption.DOT_MATCHES_ALL)
private val COMMENT_REGEX_HOTFIX_DNA = "Собран пакет yandex-direct-dna версии 0.[0-9]+.0..-.*"
    .toRegex(RegexOption.DOT_MATCHES_ALL)

@Hourglass(periodInSeconds = 2 * 60 * 60)
class HotfixMonitor @Autowired constructor(
    private val startrekSession: Session,
    private val deployHotfixRepository: DeployHotfixRepository,
    private val directAppsConfProvider: DirectAppsConfProvider,
) : DirectJob() {

    private val appByTrackerComponent: Map<String, String>
        get() = directAppsConfProvider.getDirectAppsConf()
            .associate { it.trackerComponent to it.name }

    private fun getIssuesAfter(startTime: LocalDateTime): List<Issue> {
        val timeFormatted = DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm:ss").format(startTime)

        //val query = """queue:DIRECT type:release Components: "Releases: Direct", "Releases: JavaDirect", "Releases: dna"
        val query = """queue:DIRECT type:release Components: ! "Releases: CocaineDirect"
                |Updated: >"$timeFormatted" Status: Закрыт "Sort by": Created ASC""".trimMargin()

        val issues = startrekSession.issues().find(query).toList()
        logger.info("Found {} release tickets after {} ({}); query: '{}'", issues.size, startTime, timeFormatted, query)
        return issues
    }

    private fun getHotfixComments(issue: Issue, app: String?): List<Hotfix> {
        var hotfixes : MutableList<Hotfix> = mutableListOf()
        val comments: List<Comment> = issue.comments.toList()
        val closeIssueTime = comments
            .filter { it.updatedBy.login == "ppc" }
            .find { comment -> COMMENT_REGEX.matches(comment.text.get()) }?.createdAt?.toDateTime() // Получаем даты деплоея
        val hotfixComments: List<Comment> = if (app == "dna") {  //Получаем список комментов с хотфиксами для dna
            comments
                .filter { it.createdBy.login != "ppc" }
                .filter { comment -> COMMENT_REGEX_HOTFIX_DNA.matches(comment.text.get()) }
        } else { //для java-приложений
            comments
                .filter { comment -> COMMENT_REGEX_HOTFIX.matches(comment.text.get()) }
        }
        if (app == "dna") {   // для dna считаем отдельно, т.к. они в гите и у них другой формат и ревизии в шапке
            val updateDescriptionEvents = startrekSession.events().getAll(issue.key)
                .filter { it.type == "IssueUpdated" }
                .filter { filterDnaEvents(it) }
                .toList()
            updateDescriptionEvents.forEach {event ->
                val hotfixDate = event.updatedAt.toDateTime()
                event.fields.forEach {eventFieldChange ->
                    if (eventFieldChange.field.id == "description") {
                        var hotfix = Hotfix()
                        hotfix.hotfixTicketKey = COMMENT_REGEX_TICKETS.findAll(eventFieldChange.to.toString())
                            .map { it.groupValues[1] }
                            .toList()
                            .minus(
                                COMMENT_REGEX_TICKETS.findAll(eventFieldChange.from.toString())
                                    .map { it.groupValues[1] }
                                    .toList()
                            ) // В блоке выше выбираем в изменении шапки тикет(добавляемые хотфиксы) и ревизию
                        hotfix.revisions = COMMENT_REGEX_DNA_REVISIONS.findAll(eventFieldChange.to.toString()).map { it.groupValues[1] }.toList() // мапим их на тикеты из комментов
                        hotfix.deployTicketKey = issue.key
                        hotfix.app = app
                        hotfix.toProduction = hotfixDate > closeIssueTime // Считаем, что хотфикс в прод, если он выехал после закрытия(выкладки)
                        hotfix.hotfixTime = java.time.Instant.ofEpochMilli(hotfixDate.millis).atZone(ZoneId.systemDefault()).toLocalDateTime()
                        logger.debug("Found tickets - {}, found revision - {}", hotfix.hotfixTicketKey, hotfix.revisions)
                        hotfixes.add(hotfix)
                    }
                }
            }
            return hotfixes
        }

        for (hotfixComment in hotfixComments) {
            var hotfix = Hotfix()
            val hotfixTime = hotfixComment.createdAt.toDateTime()
            hotfix.toProduction = if (closeIssueTime != null && hotfixTime > closeIssueTime) true else false
            hotfix.revisions = mutableListOf()
            hotfix.hotfixTicketKey = mutableListOf()
            hotfix.deployTicketKey = issue.key
            hotfix.hotfixTime = java.time.Instant.ofEpochMilli(hotfixTime.millis).atZone(ZoneId.systemDefault()).toLocalDateTime()
            hotfix.app = app

            COMMENT_REGEX_HOTFIX_REVISIONS_AND_TICKETS.findAll(hotfixComment.text.get()).forEach {
                hotfix.revisions.add(it.groupValues[1])
                hotfix.hotfixTicketKey.add(it.groupValues[2])
                logger.debug("Found tickets - {}, found revision - {}", hotfix.hotfixTicketKey, hotfix.revisions)
            }

            hotfixes.add(hotfix)
        }
        logger.debug("{} {} have {} hf", issue.key, issue.summary, hotfixComments.size)
        return hotfixes
    }

    private fun filterDnaEvents(event: Event?): Boolean {
        return event?.fields?.any { it.field.id == "description" }?: false

    }

    /*
    * Выбираем все деплои всех приложений при помощи стартрекового фильтра.
    * Для каждого деплоя выбираем комменты с хотфиксами(по шаблону)
    * и определяем для каждого хотфикса номер ревизии и тикет, по которому был коммит
    */
    override fun execute() {
        val monthAgo = LocalDateTime.now().minusMonths(1)
        val lastHotfixDateTime = deployHotfixRepository.getLastHotfixDate() ?: START_FETCH_TIME_FOR_EMPTY_DB
        val startFethcDateTime = if (lastHotfixDateTime < monthAgo) lastHotfixDateTime else monthAgo
        logger.info("Fetching new deploys after {}", startFethcDateTime)
        val issues = getIssuesAfter(startFethcDateTime)
        logger.info("Found {} deploys after {}", issues.size, startFethcDateTime)
        val appsHotfixCount = mutableMapOf<String, Int>()
        appByTrackerComponent.values.associateWithTo(appsHotfixCount) { 0 }
        for (issue in issues){
            val appComponent = issue.components
                .map { it.display }
                .firstOrNull { it in appByTrackerComponent }
            appComponent ?: continue
            val Hotfixes = getHotfixComments(issue, appByTrackerComponent[appComponent])
            val issueKey: String = issue.key
            val app = appByTrackerComponent[appComponent]
            logger.debug("{} - st/{} - {}", app, issueKey, Hotfixes.size)
            appsHotfixCount.merge(app.toString(), Hotfixes.size, Int::plus)
            deployHotfixRepository.addHotfixes(Hotfixes)
        }
        appsHotfixCount.forEach {
            logger.debug("Summ: {} - {}", it.key, it.value)
        }
    }
}
