package ru.yandex.direct.chassis.monitor;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.time.Duration;
import java.time.Instant;
import java.util.List;
import java.util.StringJoiner;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.function.Predicate;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;

import ru.yandex.direct.chassis.properties.YdbSettings;
import ru.yandex.direct.juggler.JugglerStatus;
import ru.yandex.direct.liveresource.LiveResourceFactory;
import ru.yandex.direct.scheduler.Hourglass;
import ru.yandex.direct.scheduler.support.DirectJob;
import ru.yandex.startrek.client.Session;
import ru.yandex.startrek.client.model.IssueCreate;
import ru.yandex.yql.YqlDataSource;

import static ru.yandex.direct.chassis.configuration.StartrekConfigurationKt.MAINTENANCE_HELPERS_TRACKER_BEAN;
import static ru.yandex.direct.chassis.util.StartrekTools.buildTable;
import static ru.yandex.direct.chassis.util.StartrekTools.messageSigner;

@Hourglass(periodInSeconds = 3600)
@ParametersAreNonnullByDefault
public class TuneReExportWorkers extends DirectJob {
    private static final Logger logger = LoggerFactory.getLogger(TuneReExportWorkers.class);
    private static final String APP = "tune-re-export-workers";
    private static final String QUERY_SOURCE = "classpath:///reexport_iterations.yql";
    private static final String SETTING_LAST_RUN_TIME = "lastRun";
    private static final String HISTORY_TAG = "tune_reexport_workers";
    private static final String HISTORY_URL = "https://st.yandex-team.ru/DIRECT/order:key:false" +
            "/filter?resolution=notEmpty()&tags=" + HISTORY_TAG;
    /**
     * Раз в сколько пересчитываем статистику и создаем тикет
     */
    private static final Duration CALCULATE_INTERVAL = Duration.ofDays(28);
    private static final String SELF = "https://a.yandex-team.ru/arc/trunk/arcadia/direct/apps" +
            "/chassis/src/main/java/ru/yandex/direct/chassis/monitor/TuneReExportWorkers.java";

    private final YqlDataSource yqlDataSource;
    private final YdbSettings ydbTool;
    private final Session tracker;
    private final String query;
    private Instant lastRun = Instant.ofEpochSecond(0);

    @Autowired
    public TuneReExportWorkers(
            @Qualifier(MAINTENANCE_HELPERS_TRACKER_BEAN) Session session,
            YdbSettings ydbSettings,
            YqlDataSource yqlDataSource
    ) {
        this.yqlDataSource = yqlDataSource;
        tracker = session;
        ydbTool = ydbSettings;

        logger.debug("load query");
        query = LiveResourceFactory.get(QUERY_SOURCE).getContent();
    }

    private void initSettings() {
        try {
            String setting = ydbTool.getSetting(APP, SETTING_LAST_RUN_TIME);

            if (setting == null) {
                logger.info("setting doesn't exists, let's set it");
                updateLastRun(lastRun);
            } else {
                logger.debug("load data from settings");
                lastRun = Instant.parse(setting);
            }
        } catch (Exception e) {
            logger.error("Failed to load settings", e);
            throw new RuntimeException(e);
        }
    }

    private boolean needWork() {
        Instant now = Instant.now();
        Duration between = Duration.between(lastRun, now);
        boolean needWork = between.compareTo(CALCULATE_INTERVAL) >= 0;
        logger.info("Last run {} was {} ago, so we need to work: {}", lastRun, between, needWork);
        return needWork;
    }

    /**
     * при недоступном хранилище - лучше упасть, и запустить YQL еще раз (тем более что там кеширование),
     * чем создать пачку лишних тикетов в трекере
     * <p>
     * вместо этого можно было смотреть на последний созданный тикет в трекере но это:
     * - не надежно (метку можно удалить)
     * - поиск по трекеру - не бесплатный (по нагрузке)
     */
    private void pingStorage() {
        Instant toDo = lastRun.plusSeconds(600);
        updateLastRun(toDo);
    }

    private void updateLastRun(Instant toDo) {
        logger.info("updating {} to {}", SETTING_LAST_RUN_TIME, toDo);
        String value = toDo.toString();
        ydbTool.setSetting(APP, SETTING_LAST_RUN_TIME, value);
    }

    private List<ResponseRow> getCurrentStat() {
        logger.info("execute YQL-query: calculating stat from messages log");
        try (Connection connection = yqlDataSource.getConnection();
             PreparedStatement stmt = connection.prepareStatement(query);
             ResultSet rs = stmt.executeQuery()) {
            logger.info("parse query result");
            return StreamEx.produce(ResponseRow.createProducer(rs))
                    .sortedBy(ResponseRow::getShard)
                    .toList();
        } catch (SQLException | RuntimeSqlException e) {
            logger.error("Failed to perform YQL-query", e);
            throw new RuntimeException(e);
        }
    }

    private void doWork() {
        List<ResponseRow> result = getCurrentStat();

        List<String> headers = List.of("шард", "текущее количество воркеров",
                "среднее время итерации", "максимальное время итерации", "количество полных итераций");
        List<Function<ResponseRow, Object>> columns = List.of(ResponseRow::getShard, ResponseRow::getCurrentWorkers,
                ResponseRow::getAvgPretty, ResponseRow::getMaxPretty, ResponseRow::getFullIterations);
        String table = buildTable(headers, result, columns);

        String shardsToIncrease = result.stream()
                .filter(r -> r.getAvg().compareTo(Duration.ofDays(4)) >= 0)
                .map(ResponseRow::getShard)
                .sorted()
                .map(Object::toString)
                .collect(Collectors.joining(", "));

        String shardsToDecrease = result.stream()
                .filter(r -> r.getAvg().compareTo(Duration.ofDays(2)) <= 0)
                .filter(r -> r.getCurrentWorkers() > 1)
                .map(ResponseRow::getShard)
                .sorted()
                .map(Object::toString)
                .collect(Collectors.joining(", "));

        String shardsToSet = result.stream()
                .filter(r -> r.getCurrentWorkers() == 0)
                .map(ResponseRow::getShard)
                .sorted()
                .map(Object::toString)
                .collect(Collectors.joining(", "));

        StringBuilder description = new StringBuilder()
                .append("В рамках этого тикета предлагается **проверить текущую статистику ре-экспорта")
                .append(" и при необходимости скорректировать количество воркеров** так")
                .append(", чтобы удержать __среднее время__ итерации на уровне 4 дней.")
                .append("\n\n")
                .append("Рекомендуемые действия:\n");

        boolean hasActions = false;
        if (!shardsToIncrease.isBlank()) {
            hasActions = true;
            description.append("* !!Добавить воркеров в шардах:!! ")
                    .append(shardsToIncrease)
                    .append('\n');
        }
        if (!shardsToSet.isBlank()) {
            hasActions = true;
            description.append("* Задать __явно__ минимальное количество (один) воркеров в шардах: ")
                    .append(shardsToSet)
                    .append('.')
                    .append('\n');
        }
        if (!shardsToDecrease.isBlank()) {
            hasActions = true;
            description.append("* Уменьшить количество воркеров в шардах: ")
                    .append(shardsToDecrease)
                    .append('\n');
        }
        if (!hasActions) {
            description.append("* Статистика выглядит хорошо, рекомендаций нет.\n");
        }

        description.append("\n\n")
                .append("((https://direct.yandex.ru/internal_tools/#set_bsexport_full_lb_export_workers_num")
                .append(" Интерфейс для настройки количества воркеров))")
                .append("\n\n")
                .append("Текущая статистика ре-экспорта по шардам:")
                .append('\n')
                .append(table)
                .append("\n\n")
                .append("((")
                .append(HISTORY_URL)
                .append(" предыдущие тикеты))")
                .append("\n\n------\n")
                .append("<{Что такое ре-экспорт и зачем этот тикет?\n")
                .append("Ре-экспорт (st:DIRECT-56895) - это фоновый процесс по переотправке всех кампаний директа")
                .append(" в контент-систему БК (только LogBroker, без BSSOAP).")
                .append(" Существует договоренность //(устная)// о том,")
                .append(" что все кампании Директа будут переотправлены за 7 дней.")
                .append(" Внутренний (для себя) target - 4 дня.")
                .append(" Отправляя данные быстрее - мы впустую греем воздух серверами")
                .append(", а медленнее - повышаем риск накопления ошибок в системе.}>\n")
                .append("<{Заметки на полях:\n")
                .append("* время итерации рассчитывается приблизительно - по моменту добавления в очередь")
                .append(" последней кампании в шарде\n")
                .append("* расчет ведется за две недели (не включая последние три дня), подробнее в YQL-запросе\n")
                .append("* максимальное время приведено в таблице для выявления поломок ре-экспорта\n")
                .append("* если среднее время не превышет 4.5 дней - увеличение воркеров можно пропустить")
                .append(", так как по опыту время итерации за следующий месяц не превысит 7 дней\n")
                .append("}>\n");

        pingStorage();

        logger.info("create new ticket");
        IssueCreate issue = IssueCreate.builder()
                .queue("DIRECT")
                .assignee("ppalex")
                .summary("Регулярный пересмотр настроек ре-экспорта")
                .description(messageSigner(this, SELF, description.toString()))
                .tags(HISTORY_TAG)
                .build();
        tracker.issues().create(issue);

        updateLastRun(Instant.now());
    }

    @Override
    public void execute() {
        initSettings();
        if (needWork()) {
            doWork();
            setJugglerStatus(JugglerStatus.OK, "successfully created new ticket");
        } else {
            setJugglerStatus(JugglerStatus.OK, "previous ticket still fresh");
        }
    }

    private static class ResponseRow {
        private final Integer shard;
        private final Integer fullIterations;
        private final Integer currentWorkers;
        private final Duration avg;
        private final Duration max;

        private ResponseRow(ResultSet rs) throws SQLException {
            shard = rs.getInt("shard");
            fullIterations = rs.getInt("full_iterations");
            currentWorkers = rs.getInt("current_workers");
            avg = Duration.ofMinutes(rs.getLong("avg"));
            max = Duration.ofMinutes(rs.getLong("max"));
        }

        public Integer getShard() {
            return shard;
        }

        public Integer getFullIterations() {
            return fullIterations;
        }

        public Duration getMax() {
            return max;
        }

        public Duration getAvg() {
            return avg;
        }

        public Integer getCurrentWorkers() {
            return currentWorkers;
        }

        public String getMaxPretty() {
            return formatDuration(max);
        }

        public String getAvgPretty() {
            return formatDuration(avg);
        }

        private static String formatDuration(Duration duration) {
            return String.format("%dд %02d:%02d",
                    duration.toDaysPart(),
                    duration.toHoursPart(),
                    duration.toMinutesPart());
        }

        static Predicate<Consumer<? super ResponseRow>> createProducer(ResultSet rs) {
            return action -> {
                try {
                    boolean hasNext = rs.next();
                    if (hasNext) {
                        action.accept(new ResponseRow(rs));
                    }
                    return hasNext;
                } catch (SQLException e) {
                    throw new RuntimeSqlException(e);
                }
            };
        }

        @Override
        public String toString() {
            return new StringJoiner(", ", ResponseRow.class.getSimpleName() + "[", "]")
                    .add("shard=" + shard)
                    .add("fullIterations=" + fullIterations)
                    .add("currentWorkers=" + currentWorkers)
                    .add("avg=" + avg)
                    .add("max=" + max)
                    .toString();
        }
    }

    private static class RuntimeSqlException extends RuntimeException {
        RuntimeSqlException(Throwable cause) {
            super(cause);
        }
    }
}
