package ru.yandex.direct.chassis.oneshot

import org.slf4j.LoggerFactory
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Component
import ru.yandex.direct.chassis.util.startrek.StartrekField
import ru.yandex.direct.chassis.util.startrek.StartrekHelper
import ru.yandex.direct.chassis.util.startrek.StartrekHelper.Companion.field
import ru.yandex.direct.chassis.util.startrek.StartrekHelper.Companion.issueUpdate
import ru.yandex.startrek.client.model.Issue

val VAL_NEED_CHECK = "Требуется проверка"
val VAL_CHECK_DONE = "Проверка сделана"

data class AppField(
    val app: String,
    val fieldName: String,
    val fieldText: String,
)

val APP_FIELDS = listOf(
    AppField("api5", "javaApiTesting", "Java-api Testing"),
    AppField("intapi", "javaIntapiTesting", "Java-intapi Testing"),
    AppField("jobs", "javaJobsTesting", "Java-jobs Testing"),
    AppField("logviewer", "javaLogviewerTesting", "Java-logviewer Testing"),
    AppField("web", "javaWebTesting", "Java-web Testing"),
)

/**
 * Миграция со старый полей "Java-xxxx Testing" в новые "Affected version" / "Tested version"
 */
@Component
class TestingFieldsConverter @Autowired constructor(
    val startrek: StartrekHelper
) : Runnable {
    private val logger = LoggerFactory.getLogger(TestingFieldsConverter::class.java)

    override fun run() {
        val fieldsCond = APP_FIELDS.map { "\"${it.fieldText}\": !empty()" }.joinToString(" or ")
        val query = """
            | Queue: DIRECT
            | and Updated: >= now() - 14days
            | and ( ${fieldsCond} )
            | and "Affected Apps": empty()
            | and "Tested Apps": empty()
            | """.trimMargin()

        logger.info("Start find query")
        val issues = startrek.session.issues().find(query).toList()
        for (issue in issues) {
            processIssue(issue)
        }
        logger.info("processed {} issues", issues.size)
    }


    private fun processIssue(issue: Issue) {
        logger.info("Process issue https://st.yandex-team.ru/{}", issue.key)
        val affected = APP_FIELDS
            .filter { issue.field<String>(it.fieldName) in listOf(VAL_NEED_CHECK, VAL_CHECK_DONE) }
            .map { it.app }
            .ifEmpty { listOf("none") }
        val tested = APP_FIELDS
            .filter { issue.field<String>(it.fieldName) == VAL_CHECK_DONE }
            .map { it.app }

        logger.info("Update Affected apps: '${affected}', Tested apps: '${tested}'")
        startrek.session.issues().update(issue, issueUpdate {
            set(StartrekField.AFFECTED_APPS, affected)
            set(StartrekField.TESTED_APPS, tested)
        }, false, false)
    }
}
