package ru.yandex.direct.chassis.properties

import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Component
import ru.yandex.direct.dbschema.chassis.Tables
import ru.yandex.direct.dbutil.SqlUtils
import ru.yandex.direct.dbutil.wrapper.DslContextProvider
import java.util.function.Consumer
import javax.annotation.ParametersAreNonnullByDefault

/**
 * Облегченная пародия на {@link PpcPropertiesSupport}
 */
@Component
@ParametersAreNonnullByDefault
class PropertiesLightSupport @Autowired constructor(
    private val dslContextProvider: DslContextProvider
) {

    /**
     * Значение ключа [name], или `null`, если такого ключа нет
     */
    operator fun get(name: String): String? {
        return getByNames(setOf(name))[name]
    }

    /**
     * Создает пару ключ-значение. Если такой ключ уже существует, значение в таблице обновляется
     */
    operator fun set(name: String, value: String?) {
        val insertStep = dslContextProvider.chassis()
            .insertInto(Tables.PROPERTIES, Tables.PROPERTIES.NAME, Tables.PROPERTIES.VALUE)
            .values(name, value)
        SqlUtils.onConflictUpdate(insertStep, setOf(Tables.PROPERTIES.VALUE))
            .execute()
    }

    private fun getByNames(names: Collection<String>): Map<String, String?> {
        if (names.isEmpty()) {
            return emptyMap()
        }
        val keysToValues = dslContextProvider.chassis()
            .select(Tables.PROPERTIES.NAME, Tables.PROPERTIES.VALUE)
            .from(Tables.PROPERTIES)
            .where(Tables.PROPERTIES.NAME.`in`(names))
            .fetchMap(Tables.PROPERTIES.NAME, Tables.PROPERTIES.VALUE)
        names.forEach(Consumer { name -> keysToValues.putIfAbsent(name, null) })
        return keysToValues
    }
}
