package ru.yandex.direct.chassis.properties;

import java.time.Instant;
import java.util.Map;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.yandex.ydb.table.description.TableDescription;
import com.yandex.ydb.table.query.Params;
import com.yandex.ydb.table.result.ResultSetReader;
import com.yandex.ydb.table.result.ValueReader;
import com.yandex.ydb.table.values.DictType;
import com.yandex.ydb.table.values.PrimitiveType;
import com.yandex.ydb.table.values.PrimitiveValue;
import com.yandex.ydb.table.values.Value;
import one.util.streamex.EntryStream;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.chassis.util.ydb.YdbClient;

import static com.google.common.base.Preconditions.checkNotNull;
import static com.yandex.ydb.table.transaction.TxControl.onlineRo;
import static com.yandex.ydb.table.transaction.TxControl.serializableRw;
import static ru.yandex.direct.chassis.util.ydb.YdbClient.KEEP_IN_CACHE;
import static ru.yandex.direct.chassis.util.ydb.YdbClient.V1_SYNTAX;

@ParametersAreNonnullByDefault
@Component
public class YdbSettings {
    private static final Logger logger = LoggerFactory.getLogger(YdbSettings.class);
    private static final String SETTINGS_TABLE = "settings";
    private static final TableDescription SETTINGS_DESCRIPTION = TableDescription.newBuilder()
            .addNullableColumn("app", PrimitiveType.utf8())
            .addNullableColumn("key", PrimitiveType.utf8())
            .addNullableColumn("value", PrimitiveType.utf8())
            .addNullableColumn("updatedAt", PrimitiveType.datetime())
            .setPrimaryKeys("app", "key")
            .build();

    private final YdbClient ydbClient;

    @Autowired
    public YdbSettings(YdbClient ydbClient) {
        this.ydbClient = ydbClient;
    }

    public void setSetting(String app, String key, String value) {
        checkNotNull(key, "key is mandatory");
        checkNotNull(value, "value can not be null");
        setSettings(app, Map.of(key, value));
    }

    public void setSettings(String app, Map<String, String> settings) {
        checkNotNull(app, "app is mandatory");
        logger.info("save app {} settings: {}", app, settings);

        String query = V1_SYNTAX +
                "DECLARE $app AS Utf8;\n" +
                "DECLARE $settings AS Dict<Utf8,Utf8>;\n" +
                "DECLARE $mt AS Datetime;\n" +
                "REPLACE INTO `" + SETTINGS_TABLE + "` " +
                "SELECT $app AS app, $mt AS updatedAt, setting.0 AS key, setting.1 AS value " +
                "FROM (SELECT $settings AS settings) " +
                "FLATTEN BY settings AS setting";

        @SuppressWarnings("rawtypes")
        var map = EntryStream.of(settings)
                .mapKeys(PrimitiveValue::utf8)
                .mapValues(PrimitiveValue::utf8)
                .mapKeys(v -> (Value) v)
                .mapValues(v -> (Value) v)
                .toMap();

        Params params = Params.of(
                "$app", PrimitiveValue.utf8(app),
                "$settings", DictType.of(PrimitiveType.utf8(), PrimitiveType.utf8()).newValueOwn(map),
                "$mt", PrimitiveValue.datetime(Instant.now())
        );

        ydbClient.supplyResult(session -> session.executeDataQuery(query, serializableRw(), params, KEEP_IN_CACHE))
                .expect("error updating settings for app " + app);
    }

    @Nullable
    public String getSetting(String app, String key) {
        logger.debug("get setting {} (app {})", key, app);

        String query = V1_SYNTAX +
                "DECLARE $app AS utf8; DECLARE $key AS utf8;\n"
                + "SELECT value FROM `" + SETTINGS_TABLE + "` WHERE app = $app AND key = $key LIMIT 1";
        Params params = Params.of(
                "$app", PrimitiveValue.utf8(app),
                "$key", PrimitiveValue.utf8(key));

        ResultSetReader rs = ydbClient.supplyResult(session -> session.executeDataQuery(query, onlineRo(), params, KEEP_IN_CACHE))
                .expect("error retrieving setting " + key)
                .getResultSet(0);

        if (rs.getRowCount() == 0) {
            return null;
        }

        rs.setRowIndex(0);
        ValueReader valueReader = rs.getColumn(0);
        if (!valueReader.isOptionalItemPresent()) {
            return null;
        }

        return valueReader.getValue().asOptional().get().asData().getUtf8();
    }

    public void createTables() {
        ydbClient.createTable(SETTINGS_TABLE, SETTINGS_DESCRIPTION);
    }
}
