package ru.yandex.direct.chassis.repository

import org.springframework.stereotype.Component
import ru.yandex.direct.chassis.entity.buildbot.BuildInfo
import ru.yandex.direct.common.util.RepositoryUtils.booleanFromLong
import ru.yandex.direct.common.util.RepositoryUtils.booleanToLong
import ru.yandex.direct.dbschema.chassis.tables.BuildbotBuilds.BUILDBOT_BUILDS
import ru.yandex.direct.dbutil.wrapper.DslContextProvider
import ru.yandex.direct.jooqmapperhelper.InsertHelper

@Component
class BuildbotBuildRepository(
    private val dslContextProvider: DslContextProvider,
) {
    fun insertBuildInfo(buildInfo: List<BuildInfo>) {
        val insertHelper = InsertHelper(dslContextProvider.chassis(), BUILDBOT_BUILDS)

        buildInfo.forEach { build ->
            insertHelper
                .set(BUILDBOT_BUILDS.BUILDER, build.builder)
                .set(BUILDBOT_BUILDS.BUILD_NUMBER, build.buildNumber)
                .set(BUILDBOT_BUILDS.REVISION, build.revision)
                .set(BUILDBOT_BUILDS.BRANCH, build.branch)
                .set(BUILDBOT_BUILDS.STATUS, build.status)
                .set(BUILDBOT_BUILDS.STATUS_TEXT, build.statusText)
                .set(BUILDBOT_BUILDS.COMMIT_MESSAGE, build.commitMessage)
                .set(BUILDBOT_BUILDS.START_TIME, build.startTime)
                .set(BUILDBOT_BUILDS.END_TIME, build.endTime)
                .set(BUILDBOT_BUILDS.IS_RESTARTED, booleanToLong(build.restarted))
                .newRecord()
        }

        insertHelper
            .onDuplicateKeyIgnore()
            .executeIfRecordsAdded()
    }

    fun getBuildsForRevision(revisions: Collection<Long>): List<BuildInfo> {
        return dslContextProvider.chassis()
            .select(
                BUILDBOT_BUILDS.ID,
                BUILDBOT_BUILDS.BUILDER,
                BUILDBOT_BUILDS.BUILD_NUMBER,
                BUILDBOT_BUILDS.REVISION,
                BUILDBOT_BUILDS.BRANCH,
                BUILDBOT_BUILDS.STATUS,
                BUILDBOT_BUILDS.STATUS_TEXT,
                BUILDBOT_BUILDS.COMMIT_MESSAGE,
                BUILDBOT_BUILDS.START_TIME,
                BUILDBOT_BUILDS.END_TIME,
                BUILDBOT_BUILDS.IS_RESTARTED,
            )
            .from(BUILDBOT_BUILDS)
            .where(BUILDBOT_BUILDS.REVISION.`in`(revisions))
            .fetch { record ->
                BuildInfo(
                    id = record[BUILDBOT_BUILDS.ID],
                    builder = record[BUILDBOT_BUILDS.BUILDER],
                    buildNumber = record[BUILDBOT_BUILDS.BUILD_NUMBER],
                    revision = record[BUILDBOT_BUILDS.REVISION],
                    branch = record[BUILDBOT_BUILDS.BRANCH],
                    status = record[BUILDBOT_BUILDS.STATUS],
                    statusText = record[BUILDBOT_BUILDS.STATUS_TEXT],
                    commitMessage = record[BUILDBOT_BUILDS.COMMIT_MESSAGE],
                    startTime = record[BUILDBOT_BUILDS.START_TIME],
                    endTime = record[BUILDBOT_BUILDS.END_TIME],
                    restarted = booleanFromLong(record[BUILDBOT_BUILDS.IS_RESTARTED]),
                )
            }
    }

    fun setRestarted(buildIds: Collection<Long>) {
        if (buildIds.isEmpty()) {
            return
        }

        dslContextProvider.chassis()
            .update(BUILDBOT_BUILDS)
            .set(BUILDBOT_BUILDS.IS_RESTARTED, booleanToLong(true))
            .where(BUILDBOT_BUILDS.ID.`in`(buildIds))
            .execute()
    }
}
