package ru.yandex.direct.chassis.repository

import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Component
import ru.yandex.direct.chassis.model.CommitDelay
import ru.yandex.direct.chassis.model.Deploy
import ru.yandex.direct.dbschema.chassis.tables.Commit.COMMIT
import ru.yandex.direct.dbschema.chassis.tables.CommitDelay.COMMIT_DELAY
import ru.yandex.direct.dbschema.chassis.tables.CommitTimings.COMMIT_TIMINGS
import ru.yandex.direct.dbutil.wrapper.DslContextProvider
import ru.yandex.direct.jooqmapperhelper.InsertHelper
import java.time.DayOfWeek
import java.time.Duration
import java.time.LocalDateTime
import java.time.ZoneOffset
import java.time.temporal.ChronoUnit.DAYS as Days

@Component
class CommitDeployRepository @Autowired constructor(private val dslContextProvider: DslContextProvider) {

    val defaultStartNightHour : Int = 20
    val defaultEndNightHour : Int = 8

    fun addDelays(commitDelays: List<CommitDelay>) {
        if (commitDelays.isEmpty()) return

        val insert = InsertHelper(dslContextProvider.chassis(), COMMIT_DELAY)
        commitDelays.forEach {
            insert
                .set(COMMIT_DELAY.APP, it.app)
                .set(COMMIT_DELAY.COMMIT_REVISION, it.commit.revision)
                .set(COMMIT_DELAY.DEPLOY_DELAY_SECONDS, it.deployDelay?.seconds)
                .set(COMMIT_DELAY.DEPLOY_DELAY_WO_WEEKEND_SECONDS, it.deployDelayWoWeekend?.seconds)
                .set(COMMIT_DELAY.DEPLOY_DELAY_WO_WEEKEND_AND_NIGHT_SECONDS, it.deployDelayWoWeekendAndNight?.seconds)
                .set(COMMIT_DELAY.DEPLOY_ID, it.deploy?.id)
                .newRecord()
        }
        insert.execute()
    }

    fun assignCorrectedDeploy(deploy: Deploy, fromExclusive: Long, toInclusive: Long, excluding: List<Long>, including: List<Long>) {
        val commitsWithLocalDateTime = getCommits(deploy, fromExclusive, toInclusive, excluding, including)
        commitsWithLocalDateTime.forEach() { commitWithLocalDateTime ->
            dslContextProvider.chassis().update(COMMIT_DELAY)
                .set(COMMIT_DELAY.DEPLOY_DELAY_SECONDS, Duration.between(commitWithLocalDateTime.value, deploy.deployTime).toSeconds())
                .set(COMMIT_DELAY.DEPLOY_DELAY_WO_WEEKEND_SECONDS, countOnlyWorkDays(commitWithLocalDateTime.value, deploy.deployTime))
                .set(COMMIT_DELAY.DEPLOY_DELAY_WO_WEEKEND_AND_NIGHT_SECONDS, countOnlyDaylightHours(commitWithLocalDateTime.value, deploy.deployTime))
                .set(COMMIT_DELAY.DEPLOY_ID, deploy.id)
                .where(COMMIT_DELAY.COMMIT_REVISION.eq(commitWithLocalDateTime.key).and(COMMIT_DELAY.APP.eq(deploy.app)))
                .execute()
            if (commitWithLocalDateTime.key !in including) {
                for (rt in listOf("Rft", "RmAcc", "Rtd")) {
                    val deployEventTimeList = deploy.javaClass.getMethod("get" + rt + "Time").invoke(deploy) as List<LocalDateTime>
                    deployEventTimeList.minByOrNull{ it > commitWithLocalDateTime.value }?.let {
                        dslContextProvider.chassis().insertInto(COMMIT_TIMINGS)
                            .set(COMMIT_TIMINGS.COMMIT_REVISION, commitWithLocalDateTime.key)
                            .set(COMMIT_TIMINGS.APP, deploy.app)
                            .set(COMMIT_TIMINGS.DELAY_SECONDS, Duration.between(commitWithLocalDateTime.value, it).toSeconds())
                            .set(COMMIT_TIMINGS.DELAY_WO_WEEKEND_SECONDS, countOnlyWorkDays(commitWithLocalDateTime.value, it))
                            .set(COMMIT_TIMINGS.DELAY_WO_WEEKEND_AND_NIGHT_SECONDS, countOnlyDaylightHours(commitWithLocalDateTime.value, it))
                            .set(COMMIT_TIMINGS.DELAY_TYPE, rt)
                            .set(COMMIT_TIMINGS.DEPLOY_ID, deploy.id)
                            .onDuplicateKeyIgnore()
                            .execute()
                    }
                }
            }
        }
    }

    fun getCommits(deploy: Deploy, fromExclusive: Long, toInclusive: Long, excluding: List<Long> = listOf(), including: List<Long> = listOf()) : Map<Long, LocalDateTime> {
        val commitsWithLocalDateTime : Map<Long, LocalDateTime> =
            dslContextProvider.chassis().select(COMMIT.REVISION, COMMIT.TIME).from(COMMIT)
                .leftJoin(COMMIT_DELAY).on(COMMIT.REVISION.eq(COMMIT_DELAY.COMMIT_REVISION))
                .where(COMMIT_DELAY.APP.eq(deploy.app).and(COMMIT_DELAY.COMMIT_REVISION.greaterThan(fromExclusive).and(COMMIT_DELAY.COMMIT_REVISION.lessOrEqual(toInclusive)).or(COMMIT_DELAY.COMMIT_REVISION.`in`(including)))
                    .and(COMMIT_DELAY.COMMIT_REVISION.notIn(excluding))).fetchMap(COMMIT.REVISION, COMMIT.TIME)
        return commitsWithLocalDateTime
    }

    fun countOnlyWorkDays(startDateTime: LocalDateTime, endDateTime: LocalDateTime): Long {
        if ((startDateTime.toLocalDate() == endDateTime.toLocalDate()) and (isHoliday(startDateTime.dayOfWeek))) //Если все произошло в 1 день и он выходной
            return 0
        var secondsBetween: Long = endDateTime.toEpochSecond(ZoneOffset.UTC).minus(startDateTime.toEpochSecond(ZoneOffset.UTC))
        var daysBetween: Long = Days.between(startDateTime.toLocalDate().atStartOfDay().plusDays(1), endDateTime.toLocalDate().atStartOfDay())

        if (isHoliday(startDateTime.dayOfWeek)) { //Начало попало на выходной
            val startDateSeconds : Int = (24 * 3600 - startDateTime.toLocalTime().toSecondOfDay())
            secondsBetween -= startDateSeconds
        }

        if (daysBetween > 7) {
            val countWeek: Long = daysBetween.div(7)
            secondsBetween -= countWeek * 2 * 24 * 3600
            daysBetween = daysBetween.rem(7)
        }

        for (i in 1..daysBetween ) if (isHoliday(startDateTime.plusDays(i).dayOfWeek)) {
            secondsBetween -= 24 * 3600
        }

        if (isHoliday(endDateTime.dayOfWeek)) { //Конец попал на выходной
            secondsBetween -= endDateTime.toLocalTime().toSecondOfDay()
        }

        return secondsBetween
    }

    private fun isHoliday(dayOfWeek: DayOfWeek): Boolean {
        return dayOfWeek in sequenceOf(DayOfWeek.SATURDAY, DayOfWeek.SUNDAY)
    }

    fun countOnlyDaylightHours(startDateTime: LocalDateTime, endDateTime: LocalDateTime, startNight: Int = defaultStartNightHour, endNight: Int = defaultEndNightHour): Long {
        var workTime: Long = endDateTime.toEpochSecond(ZoneOffset.UTC) - startDateTime.toEpochSecond(ZoneOffset.UTC)
        if ((startDateTime.dayOfYear == endDateTime.dayOfYear) and
            (startDateTime.year == endDateTime.year)) {
            if (isHoliday(startDateTime.dayOfWeek)) return 0 //Все произошло в выходной
            when (startDateTime.hour) {
                in startNight..23 -> return 0 //в один день, вечером началось
                in 0..(endNight - 1) -> workTime -= (endNight * 3600 - startDateTime.toLocalTime().toSecondOfDay()) // С утра, выкидывааем кусок до рабочего времени
            }
            when (endDateTime.hour) {
                in 0..(endNight - 1) -> return 0 //в один день, еще утром закончилось
                in startNight..23 -> workTime -= ((24 - startNight) * 3600 - (86400 - endDateTime.toLocalTime().toSecondOfDay())) //Вечером, выкидываем кусок от начала ночи до события(вычитаем от события до конца суток)
            }
        }
        else {
            when (startDateTime.hour) {
                in 0..(endNight - 1) -> {
                    workTime -= (((24 - startNight) + endNight) * 3600 - startDateTime.toLocalTime().toSecondOfDay()) //Выкидываем весь вечер и кусок от начала до рабочего дня
                    if (isHoliday(startDateTime.dayOfWeek)) workTime -= (startNight - endNight) * 3600 //Если выходной, то и все рабочие часы
                }
                in startNight..23 -> workTime -= (86400 - startDateTime.toLocalTime().toSecondOfDay()) //Выкидываем кусок от события до конца суток
                else -> {
                    workTime -= (24 - startNight) * 3600 //Выкидываем точно вечернюю часть
                    if (isHoliday(startDateTime.dayOfWeek)) workTime -= (startNight * 3600 - startDateTime.toLocalTime().toSecondOfDay()) //Выкидываем часть от старта до начала ночи
                }

            }
            when (endDateTime.hour) {
                in 0..(endNight - 1) -> workTime -= endDateTime.toLocalTime().toSecondOfDay() //Выкидываем все
                in startNight..23 -> {
                    workTime -= (endDateTime.toLocalTime().toSecondOfDay() + (endNight - startNight) * 3600) // Выкидываем утро и часть от начала ночи до события
                    if (isHoliday(startDateTime.dayOfWeek)) workTime -= (startNight - endNight) * 3600 //Выкидываем и рабочие часы
                }
                else -> {
                    if (isHoliday(startDateTime.dayOfWeek))
                        workTime -= endDateTime.toLocalTime().toSecondOfDay() //Выкидываем весь день
                    else
                        workTime -= endNight * 3600 //Выкидываем утро
                }
            }

            var daysBetween: Long = Days.between(startDateTime.plusDays(1).toLocalDate().atStartOfDay(), endDateTime.toLocalDate().atStartOfDay())
            workTime -= daysBetween * (endNight + (24 - startNight)) * 3600 //Выкидываем все ночи промежуточных дней
            if (daysBetween > 7) {
                val countWeek: Long = daysBetween.div(7)
                workTime -= countWeek * 2 * (startNight - endNight) * 3600
                daysBetween = daysBetween.rem(7)
            }
            for (i in 1..daysBetween ) if (isHoliday(startDateTime.plusDays(i).dayOfWeek)) {
                workTime -= (startNight - endNight) * 3600 //Выкидываем рабочие часы выходных
            }

        }
        return workTime
    }
}
