package ru.yandex.direct.chassis.repository

import org.jooq.Record
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Component
import ru.yandex.direct.chassis.model.Deploy
import ru.yandex.direct.dbschema.chassis.tables.Deploy.DEPLOY
import ru.yandex.direct.dbutil.wrapper.DslContextProvider
import ru.yandex.direct.jooqmapperhelper.InsertHelper
import java.time.LocalDateTime
import javax.annotation.ParametersAreNonnullByDefault

@Component
@ParametersAreNonnullByDefault
class DeployRepository @Autowired constructor(private val dslContextProvider: DslContextProvider) {

    fun addDeploys(deploys: List<Deploy>) {
        if (deploys.isEmpty()) return

        val insert = InsertHelper(dslContextProvider.chassis(), DEPLOY)
        deploys.forEach {
            insert
                .set(DEPLOY.VERSION, it.version)
                .set(DEPLOY.APP, it.app)
                .set(DEPLOY.TICKET_KEY, it.ticketKey)
                .set(DEPLOY.DEPLOY_TIME, it.deployTime)
                .set(DEPLOY.RFT_TIME, it.rftTime.joinToString())
                .set(DEPLOY.RMACC_TIME, it.rmAccTime.joinToString())
                .set(DEPLOY.RTD_TIME, it.rtdTime.joinToString())
                .set(DEPLOY.DELAY_COUNTED, if (it.delayCounted) 1L else 0L)
                .newRecord()
        }

        insert.onDuplicateKeyIgnore().execute()
    }

    fun getLastByDeployTime(): Deploy? {
        return dslContextProvider.chassis()
            .select(DEPLOY.ID, DEPLOY.VERSION, DEPLOY.APP, DEPLOY.TICKET_KEY, DEPLOY.RFT_TIME, DEPLOY.RMACC_TIME, DEPLOY.RTD_TIME, DEPLOY.DEPLOY_TIME, DEPLOY.DELAY_COUNTED)
            .from(DEPLOY)
            .orderBy(DEPLOY.DEPLOY_TIME.desc())
            .limit(1)
            .fetchOne()?.map(this::createDeploy)
    }

    fun getDeploysForApp(app: String): List<Deploy> {
        return dslContextProvider.chassis()
            .select(DEPLOY.ID, DEPLOY.VERSION, DEPLOY.APP, DEPLOY.TICKET_KEY, DEPLOY.RFT_TIME, DEPLOY.RMACC_TIME, DEPLOY.RTD_TIME, DEPLOY.DEPLOY_TIME, DEPLOY.DELAY_COUNTED)
            .from(DEPLOY)
            .where(DEPLOY.APP.eq(app))
            .fetch(this::createDeploy)
    }

    fun setDelayCounted(deploy: Deploy) {
        dslContextProvider.chassis()
            .update(DEPLOY)
            .set(DEPLOY.DELAY_COUNTED, 1)
            .where(DEPLOY.ID.eq(deploy.id))
            .execute()
    }

    private fun createDeploy(record: Record) = Deploy()
        .withId(record.get(DEPLOY.ID))
        .withApp(record.get(DEPLOY.APP))
        .withVersion(record.get(DEPLOY.VERSION))
        .withTicketKey(record.get(DEPLOY.TICKET_KEY))
        .withRftTime(record.get(DEPLOY.RFT_TIME).toString().split(',').map {  LocalDateTime.parse(it.strip()) })
        .withRmAccTime(record.get(DEPLOY.RMACC_TIME).toString().split(',').map {  LocalDateTime.parse(it.strip()) })
        .withRtdTime(record.get(DEPLOY.RTD_TIME).toString().split(',').map {  LocalDateTime.parse(it.strip()) })
        .withDeployTime(record.get(DEPLOY.DEPLOY_TIME))
        .withDelayCounted(record.get(DEPLOY.DELAY_COUNTED) > 0)

}
