package ru.yandex.direct.chassis.util

import org.slf4j.LoggerFactory
import org.springframework.stereotype.Component
import ru.yandex.yp.YpRawClient
import ru.yandex.yp.client.api.Autogen
import ru.yandex.yp.client.api.TDeployUnitStatus
import ru.yandex.yp.client.api.TStageStatus
import ru.yandex.yp.client.pods.TSandboxResource
import ru.yandex.yp.model.YpGetManyStatement
import ru.yandex.yp.model.YpGetStatement
import ru.yandex.yp.model.YpObjectType
import ru.yandex.yp.model.YpPayload
import ru.yandex.yp.model.YpSelectStatement
import ru.yandex.yp.model.YpTypedId
import java.time.Instant
import java.time.temporal.ChronoUnit
import java.util.concurrent.ExecutionException
import java.util.concurrent.TimeUnit

@Component
class DeployService(
    private val ypClientCrossDC: YpRawClient
) {
    private val logger = LoggerFactory.getLogger(DeployService::class.java)
    fun getDeployStatusSummary(app: DirectAppsConfEntry, stageName: String, stageLabel: String?): DeployStatusSummary {
        val deployUnitStatus = getDeployUnitStatus(stageName)

        if (app.yaDeployResourceType == "FRONTEND_DOCKER_IMAGE") {
            val deployBoxName = app.yaDeployBoxName[stageLabel] ?: throw MissingStage("Missing $stageLabel deploy-box-name for app ${app.name}")

            val deployDockerImageTag = deployUnitStatus.currentTarget?.imagesForBoxesMap?.get(deployBoxName)?.tag ?: ""

            logger.info("Deploy docker image tag value: $deployDockerImageTag")

            val deployTicket = getDeployTicketForVersion(stageName) { release ->
                release.spec.docker.imagesList
                    .mapNotNull { resource -> resource.tag }
                    .contains(deployDockerImageTag)
            }
            if (deployTicket != null) logger.info("Found deploy ticket ${deployTicket.meta.id} with name ${deployTicket.meta.name}")
            else logger.info("Deploy ticket is not found")

            val version = DirectReleaseUtils.getReleaseVersionFromDockerImageTag(deployDockerImageTag) ?: ""

            return DeployStatusSummary(
                app = app,
                version = version,
                deployTicket?.spec?.title,
                deployTicket?.status?.progress?.closed?.status == ru.yandex.yp.client.api.EConditionStatus.CS_TRUE,
                isWaitingForApprove(deployUnitStatus),
                Instant.ofEpochMilli(deployUnitStatus.targetSpecTimestamp),
            )
        } else {
            val resourceType = app.yaDeployResourceType ?: throw InvalidAppConfig("missing Ya.Deploy resource type")

            val sandboxResource = getSandboxResources(deployUnitStatus, resourceType)
            val sandboxResourceVersion = sandboxResource.attributesMap?.get("resource_version")!!
            val deployTicket = getDeployTicketForVersion(stageName) { release ->
                release.spec.sandbox.resourcesList
                    .mapNotNull { resource -> resource.attributesMap["resource_version"] }
                    .contains(sandboxResourceVersion)
            }
            if (deployTicket != null) logger.info("Found deploy ticket ${deployTicket.meta.id} with name ${deployTicket.meta.name}")
            else logger.info("Deploy ticket is not found")

            return DeployStatusSummary(
                app = app,
                version = sandboxResourceVersion,
                deployTicket?.spec?.title,
                deployTicket?.status?.progress?.closed?.status == ru.yandex.yp.client.api.EConditionStatus.CS_TRUE,
                isWaitingForApprove(deployUnitStatus),
                Instant.ofEpochMilli(deployUnitStatus.targetSpecTimestamp),
            )
        }
    }

    private fun isWaitingForApprove(deployUnitStatus: TDeployUnitStatus): Boolean {
        if (deployUnitStatus.latestDeployedRevision != deployUnitStatus.targetRevision) {
            if (deployUnitStatus.currentTarget.deploySettings.clusterSequenceList.any { it.needApproval }) {
                // Уже задеплоились все поды, которые могли задеплоиться до апрува
                if (deployUnitStatus.progress.podsReady == deployUnitStatus.progress.podsTotal) {
                    return true
                }
            }
        }
        return false
    }

    fun getDeployTicketForVersion(stageName: String, filterByVersion: (Autogen.TRelease) -> Boolean): Autogen.TDeployTicket? {
        val creationTimeMinMicroseconds = Instant.now().minus(14, ChronoUnit.DAYS).epochSecond * 1_000_000
        val deployTicketSelectStatement = YpSelectStatement.protobufBuilder(YpObjectType.DEPLOY_TICKET)
            .addSelector("")
            .setFilter("[/meta/stage_id]='$stageName' AND [/meta/creation_time] > $creationTimeMinMicroseconds")
            .build()
        val deployTicketsByReleaseId = ypClientCrossDC.objectService()
            .selectObjects(deployTicketSelectStatement) { p -> getDeployTickerStatus(p) }
            .get(10, TimeUnit.SECONDS)
            .results
            .filterNotNull()
            .associateBy { it.spec.releaseId }

        val releases = if (deployTicketsByReleaseId.isNotEmpty()) {
            val ypGetReleaseStatement = YpGetManyStatement.protobufBuilder(YpObjectType.RELEASE).addIds(deployTicketsByReleaseId.keys.toList()).addSelector("").build()!!
            ypClientCrossDC.objectService().getObjects(ypGetReleaseStatement, this::parseRelease).get(10, TimeUnit.SECONDS).filterNotNull()
        } else listOf()
        return releases.filter { filterByVersion(it) }.map { deployTicketsByReleaseId[it.meta.id] }.firstOrNull()

    }

    fun getDeployUnitStatus(stageName: String): TDeployUnitStatus {
        val ypGetStatement = YpGetStatement.protobufBuilder(YpTypedId(stageName, YpObjectType.STAGE)).addSelector("/status").build()!!
        try {
            return ypClientCrossDC.objectService().getObject(ypGetStatement, this::parseDeployUnitStatus).get()
        } catch (e: ExecutionException) {
            throw if (e.cause is InvalidDeployStatus) e.cause as InvalidDeployStatus else e
        }
    }

    private fun getSandboxResources(deployUnitStatus: TDeployUnitStatus, resourceType: String): TSandboxResource {
        val podTemplateSpec = if (deployUnitStatus.hasMultiClusterReplicaSet()) {
            deployUnitStatus.currentTarget?.multiClusterReplicaSet?.replicaSet?.podTemplateSpec
        } else {
            deployUnitStatus.currentTarget?.replicaSet?.replicaSetTemplate?.podTemplateSpec
        }
        return podTemplateSpec?.spec?.podAgentPayload?.spec?.resources?.layersList
            .orEmpty()
            .asSequence()
            .mapNotNull { it.meta?.sandboxResource }
            .filter { it.resourceType == resourceType }
            .filter { it.attributesMap?.containsKey("resource_version") == true }
            .firstOrNull()
            ?: throw InvalidDeployStatus("no resource with type $resourceType and resource_version attribute in layers")

    }

    private fun parseRelease(results: List<YpPayload>?): Autogen.TRelease? {
        if (results != null && results.isNotEmpty()) {
            return Autogen.TRelease.parseFrom(results[0].protobuf.get())!!
        }
        return null
    }

    private fun parseDeployUnitStatus(results: List<YpPayload>?): TDeployUnitStatus {
        if (results != null && results.isNotEmpty()) {
            val stageStatus: TStageStatus = TStageStatus.parseFrom(results[0].protobuf.get())!!
            if (stageStatus.deployUnitsCount > 1) throw InvalidDeployStatus("more than 1 deploy units")
            return stageStatus.deployUnitsMap.orEmpty().values.firstOrNull()
                ?: throw InvalidDeployStatus("no deploy units")
        } else {
            throw InvalidDeployStatus("YP API request returned no results")
        }
    }

    private fun getDeployTickerStatus(results: List<YpPayload>?): Autogen.TDeployTicket? {
        if (results != null && results.isNotEmpty()) {
            return Autogen.TDeployTicket.parseFrom(results[0].protobuf.get())!!
        }
        return null
    }
}

data class DeployStatusSummary(
    val app: DirectAppsConfEntry,
    val version: String,
    val deployTicketTitle: String?,
    val isDeployTicketClosed: Boolean,
    // Ожидает ли текущая выкладка подтверждения локации
    val isDeployWaitingForApproveLocation: Boolean,
    // Время последнего изменения спеки (по сути, момента старта последней выкладки)
    val deployTargetSpecTimestamp: Instant,
)

open class DeployException(message: String) : Exception(message)
class InvalidDeployStatus(message: String) : DeployException(message)  // может означать как невалидный ответ API, так и неправильно/неожиданно сконфигурированный сервис

class MissingStage(message: String) : DeployException(message)
class InvalidAppConfig(message: String) : DeployException(message)

enum class DeployStage(val label: String) {
    PRODUCTION("production"),
    TESTING("testing"),
    DEVTEST("devtest"),
    DEV7("dev7"),
    ;

    companion object {
        private val byLabel: Map<String, DeployStage> =
            values().associateBy { it.label }

        fun getByLabel(label: String): DeployStage = byLabel[label]
            ?: throw IllegalArgumentException("Invalid stage label: $label")
    }
}

val DEV_DEPLOY_STAGES = setOf(DeployStage.DEV7, DeployStage.DEVTEST)
