package ru.yandex.direct.chassis.util

import ru.yandex.direct.liveresource.LiveResourceEvent
import ru.yandex.direct.liveresource.LiveResourceListener
import ru.yandex.direct.utils.YamlUtils

object DirectAppsConfIgnoredFeatures {
    val RELEASES_FOR_DEPLOY_NOTIFICATION: String = "releases-for-deploy-notification"
    val DIRECT_RELEASE: String = "direct-release"
}

data class DirectAppsConfEntry(
    val name: String,
    val packageName: String?,
    val trackerAffectedApp: String?,
    val deployType: String?,
    val ignoredFeatures: List<String>,
    val trackerComponent: String,
    val trackerDeployedTag: String?,
    val yaDeployResourceType: String?,
    val yaDeployStages: Map<String, List<String>>,
    val yaDeployBoxName: Map<String, String?>,
    val newCiGraph: String?,
    val zookeeperVersionNode: String?,
) {
    val stages: Map<DeployStage, List<String>> =
        yaDeployStages.mapKeys { (label, _) -> DeployStage.getByLabel(label) }
}

class DirectAppsConfProvider(
    directAppsConfContent: String,
) : LiveResourceListener {

    private var directAppsConf: List<DirectAppsConfEntry>

    init {
        directAppsConf = parseDirectAppsConf(directAppsConfContent)
    }

    fun getDirectAppsConf(): List<DirectAppsConfEntry> {
        return directAppsConf
    }

    override fun update(event: LiveResourceEvent) {
        directAppsConf = parseDirectAppsConf(event.currentContent)
    }

    private fun parseDirectAppsConf(content: String): List<DirectAppsConfEntry> {
        val root = YamlUtils.getObjectMapper().readTree(content)
        return root["apps"].fields().asSequence().map { (name, value) ->
            val packageName: String? =
                value["package"].textValue()
            val deployType: String? =
                value["deploy_type"].textValue()
            val trackerAffectedApp =
                value["tracker-affected-app"].textValue()
            val trackerComponent =
                value["tracker-component"].textValue()
            val ignoredFeatures =
                value["ignore-features"]?.toList().orEmpty().map { it.textValue() }
            val trackerDeployedTag: String? =
                value["tracker-deployed-tag"].textValue()
            val yaDeployResourceType: String? =
                value["yadeploy-resource-type"]?.textValue()
            val yaDeployStages: Map<String, List<String>> =
                value["yadeploy-stages"]?.fields()?.asSequence().orEmpty().mapNotNull {
                    val stagesList: List<String> =
                        it.value.toList().mapNotNull { data -> data.findValue("stage")?.textValue() }

                    if (stagesList.isEmpty()) {
                        if (it.value.textValue() != null) {
                            it.key to listOf(it.value.textValue())
                        } else {
                            null
                        }
                    } else {
                        it.key to stagesList
                    }
                }.toMap()

            val yaDeployBoxName: Map<String, String?> =
                value["yadeploy-box-name"]?.fields()?.asSequence().orEmpty()
                    .map { it.key to it.value.textValue() }
                    .toMap()
            val newCiGraph: String? =
                value["newci_graph"]?.textValue()
            val zookeeperVersionNode: String? =
                value["zookeeper-version-node"].textValue()

            DirectAppsConfEntry(
                name,
                packageName,
                trackerAffectedApp,
                deployType,
                ignoredFeatures,
                trackerComponent,
                trackerDeployedTag,
                yaDeployResourceType,
                yaDeployStages,
                yaDeployBoxName,
                newCiGraph,
                zookeeperVersionNode,
            )
        }.toList()
    }
}
