package ru.yandex.direct.chassis.util

import com.trilead.ssh2.Connection
import org.apache.commons.io.IOUtils
import org.slf4j.LoggerFactory
import org.springframework.beans.factory.annotation.Value
import org.springframework.stereotype.Component
import ru.yandex.direct.liveresource.LiveResource
import ru.yandex.direct.liveresource.provider.LiveResourceFactoryBean
import ru.yandex.direct.utils.JsonUtils
import java.nio.charset.StandardCharsets

@Component
class DistUtils(
    @Value("\${chassis.dist_svn.user}")
    private val user: String,
    @Value("\${chassis.dist_svn.key}")
    key: String,
    liveResourceFactoryBean: LiveResourceFactoryBean,
) {

    private val logger = LoggerFactory.getLogger(DistUtils::class.java)

    private val keyResource: LiveResource = liveResourceFactoryBean.get(key)

    fun getPackageEnvironment(packageName: String, version: String): String? {
        logger.info("Trying to get package $packageName=$version stage from dist")
        return try {
            Utils.retryThrowing(5) {
                val output = executeDistCommand("find_package -j ${packageName}_$version")
                    .replace("\n", "")
                logger.info("Dist response for package ${packageName}_$version: $output")

                val stage = JsonUtils.fromJson(output)["result"][0]["environment"].textValue()
                logger.info("Got stage $stage for package $packageName=$version")

                return@retryThrowing stage
            }
        } catch (e: Exception) {
            logger.warn("Failed to get package version from dist", e)
            return null
        }
    }

    private fun executeDistCommand(command: String): String {
        val connection = Connection("dupload.dist.yandex.ru")
        connection.connect()
        connection.authenticateWithPublicKey(user, keyResource.content.toCharArray(), null)
        val session = connection.openSession()
        session.execCommand(command)
        return IOUtils.toString(session.stdout, StandardCharsets.UTF_8)
    }
}
