package ru.yandex.direct.chassis.util

import ru.yandex.startrek.client.model.Issue

object DirectReleaseUtils {

    fun getReleaseApp(release: Issue, apps: Collection<DirectAppsConfEntry>): DirectAppsConfEntry? {
        for (component in release.components) {
            val app = apps.find { it.trackerComponent == component.display }
            if (app != null) {
                return app
            }
        }
        return null
    }

    private val DIRECT_ISSUE_REGEX = "DIRECT-\\d+".toRegex()

    /**
     * Находит все тикеты, которые будут выложены с релизом [release]
     */
    fun getReleaseTickets(release: Issue): List<String> {
        val issueKeys = mutableListOf<String>()
        issueKeys += DIRECT_ISSUE_REGEX
            .findAll(release.description.orElse(""))
            .map { it.value }
        issueKeys += release.comments.toList().nonCf().flatMap { comment ->
            DIRECT_ISSUE_REGEX
                .findAll(comment.text.orElse(""))
                .map { it.value }
        }
        return issueKeys
    }

    fun findUnitTestIssues(release: Issue): List<Issue> {
        return release.links
            .map { it.`object` }
            .filter { it.display.contains("запуск юнит-тестов в релизе", ignoreCase = true) }
            .filter { it.display.contains(release.key, ignoreCase = true) }
            .map { it.load() }
    }

    fun findRegressionIssues(release: Issue): List<Issue> {
        return release.links
            .map { it.`object` }
            .filter { it.display.contains("запуск и анализ регрессионных автотестов", ignoreCase = true) }
            .filter { it.display.contains(release.key, ignoreCase = true) }
            .map { it.load() }
    }

    private val RELEASE_SUMMARY_REGEX =
        """^релиз(?: +[a-z0-9\-]+)?: .*? - выложить +(1\.[0-9]+(?:\.[0-9]+)?-[0-9]+)(.*)""".toRegex()
    private val RELEASE_SUMMARY_REGEX_NEWCI =
        """^релиз(?: +[a-z0-9\-]+)?: .*? NewCI v(\d+(?:\.\d+)?) - выложить +(1\.[0-9]+(?:\.[0-9]+)?-[0-9]+)(.*)""".toRegex()
    private val RELEASE_VERSION_FROM_DOCKER_IMAGE_TAG =
        """(\d+(?:\.\d+)*)-(?:[\w\d]+)""".toRegex()

    fun getReleaseVersionString(summary: String): String? {
        return RELEASE_SUMMARY_REGEX_NEWCI.matchEntire(summary)?.groupValues?.get(1)
            ?: RELEASE_SUMMARY_REGEX.matchEntire(summary)?.groupValues?.get(1)
    }

    fun getReleaseVersionFromDockerImageTag(tag: String): String? {
        return RELEASE_VERSION_FROM_DOCKER_IMAGE_TAG.matchEntire(tag)?.groupValues?.get(1)
    }

    fun getReleaseVersion(release: Issue): ReleaseVersion? {
        val versionString = getReleaseVersionString(release.summary)
            ?: return null
        return ReleaseVersion.parse(versionString)
    }
}

/**
 * Для новой схемы версионирования релизов (newci):
 * - [baseRevision] &mdash; номер релиза
 * - [headRevision] &mdash; номер хотфикса (либо `null`, если хотфиксов не было)
 *
 * Для старой схемы версионирования релизов:
 * - [baseRevision] &mdash; базовая ревизия
 * - [headRevision] &mdash; ревизия коммита в svn ветке (либо `null`, если ветки нет)
 */
data class ReleaseVersion(val baseRevision: Long, val headRevision: Long?) : Comparable<ReleaseVersion> {

    val isCi = baseRevision < 10000

    override fun compareTo(other: ReleaseVersion): Int = RELEASE_VERSION_COMPARATOR.compare(this, other)

    override fun toString(): String = when {
        isCi && headRevision == null -> "$baseRevision"
        isCi -> "$baseRevision.$headRevision"
        headRevision == null -> "1.$baseRevision-1"
        else -> "1.$baseRevision.$headRevision-1"
    }

    companion object {
        private val RELEASE_VERSION_COMPARATOR: Comparator<ReleaseVersion> = Comparator
            .comparingLong<ReleaseVersion> { it.baseRevision }
            // -1 потому что версия без хотфиксов должна быть меньше версии с хотфиксами
            .thenComparingLong { it.headRevision ?: -1 }

        private val RELEASE_VERSION_REGEX = """1\.(\d+)(?:\.(\d+))?-\d+""".toRegex()
        private val RELEASE_VERSION_REGEX_NEWCI = """(\d+)(?:\.(\d+))?""".toRegex()

        fun parse(version: String): ReleaseVersion? {
            val match = RELEASE_VERSION_REGEX.matchEntire(version)
                ?: RELEASE_VERSION_REGEX_NEWCI.matchEntire(version)
                ?: return null

            return ReleaseVersion(
                match.groups[1]!!.value.toLong(),
                match.groups[2]?.value?.toLong(),
            )
        }
    }
}
