package ru.yandex.direct.chassis.util;

import java.io.IOException;
import java.time.temporal.ChronoUnit;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.StringJoiner;
import java.util.function.Function;

import javax.annotation.ParametersAreNonnullByDefault;

import org.apache.http.client.HttpClient;
import org.apache.http.client.config.CookieSpecs;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.config.Registry;
import org.apache.http.config.RegistryBuilder;
import org.apache.http.conn.HttpClientConnectionManager;
import org.apache.http.conn.socket.ConnectionSocketFactory;
import org.apache.http.conn.socket.PlainConnectionSocketFactory;
import org.apache.http.conn.ssl.SSLConnectionSocketFactory;
import org.apache.http.impl.client.DefaultHttpRequestRetryHandler;
import org.apache.http.impl.client.HttpClientBuilder;
import org.apache.http.impl.conn.PoolingHttpClientConnectionManager;
import org.apache.http.protocol.HttpContext;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.direct.utils.ThreadUtils;
import ru.yandex.startrek.client.Session;
import ru.yandex.startrek.client.StartrekClientBuilder;
import ru.yandex.startrek.client.model.CommentCreate;
import ru.yandex.startrek.client.model.Field;

import static com.google.common.base.Preconditions.checkState;
import static ru.yandex.direct.chassis.app.ChassisApp.SERVICE;

@ParametersAreNonnullByDefault
public class StartrekTools {
    private static final Logger logger = LoggerFactory.getLogger(StartrekTools.class);

    private StartrekTools() {
    }

    public static Session createClient(String apiUrl, String oauthToken, Map<String, Field.Schema> customFields) {
        var builder = StartrekClientBuilder.newBuilder()
                .uri(apiUrl)
                .maxConnections(5)
                .httpClient(buildClient())
                .customFields(Cf.wrap(customFields));

        return builder.build(oauthToken);
    }

    public static Map<String, Field.Schema> mergeCustomFields(Collection<Map<String, Field.Schema>> customFields) {
        Map<String, Field.Schema> result = new HashMap<>();
        for (var map : customFields) {
            for (Map.Entry<String, Field.Schema> entry : map.entrySet()) {
                Field.Schema oldValue = result.put(entry.getKey(), entry.getValue());
                if (oldValue != null && !oldValue.toString().equals(entry.getValue().toString())) {
                    // у типов трекера нет нормального equals/hashCode, поэтому сравниваем строками
                    throw new IllegalStateException("Can't merge different schemas for key " + entry.getKey());
                }
            }
        }
        return Collections.unmodifiableMap(result);
    }

    private static HttpClient buildClient() {
        Registry<ConnectionSocketFactory> registry = RegistryBuilder.<ConnectionSocketFactory>create()
                .register("http", PlainConnectionSocketFactory.getSocketFactory())
                .register("https", SSLConnectionSocketFactory.getSocketFactory())
                .build();

        HttpClientConnectionManager connManager = new PoolingHttpClientConnectionManager(registry);

        RequestConfig requestConfig = RequestConfig.custom()
                .setConnectionRequestTimeout(40000)
                .setConnectTimeout(5000)
                .setSocketTimeout(40000)
                .setCookieSpec(CookieSpecs.DEFAULT)
                .build();

        return HttpClientBuilder.create()
                .setUserAgent(SERVICE)
                .setDefaultRequestConfig(requestConfig)
                .setMaxConnPerRoute(5)
                .setMaxConnTotal(5)
                .setRetryHandler(new RetryHandler(3))
                .setConnectionManager(connManager)
                .build();
    }

    public static <T> String buildTable(Collection<String> headers, Collection<T> rows,
                                        Collection<Function<T, Object>> columnExtractors) {
        checkState(headers.size() == columnExtractors.size(), "columns count doesn't match to header");

        StringBuilder sb = new StringBuilder();
        sb.append("#|");
        sb.append("\n");

        StringJoiner headerJoiner = tableRowJoiner();
        headers.forEach(headerJoiner::add);
        sb.append(headerJoiner);
        sb.append("\n");

        for (T row : rows) {
            StringJoiner rowJoiner = tableRowJoiner();
            for (Function<T, Object> columnExtractor : columnExtractors) {
                Object column = columnExtractor.apply(row);
                rowJoiner.add(column == null ? "" : column.toString());
            }
            sb.append(rowJoiner.toString());
            sb.append("\n");
        }

        sb.append("|#");
        sb.append("\n");
        return sb.toString();
    }

    public static CommentCreate.Builder signedMessageFactory(Object self, String repo, String message) {
        return CommentCreate.comment(messageSigner(self, repo, message));
    }

    public static String messageSigner(Object self, String repo, String message) {
        var logs = Utils.INSTANCE.currentLogViewerLink();
        return message
                + "\n" + "----\n" +
                "((" + repo + " " + self.getClass().getCanonicalName() + "))" +
                ", ((" + logs + " логи))";
    }

    private static StringJoiner tableRowJoiner() {
        return new StringJoiner(" | ", "|| ", " ||");
    }

    static class RetryHandler extends DefaultHttpRequestRetryHandler {
        protected RetryHandler(int retryCount) {
            super(retryCount, false, Collections.emptyList());
        }

        @Override
        public boolean retryRequest(IOException exception, int executionCount, HttpContext context) {
            ThreadUtils.sleep(5, ChronoUnit.SECONDS);
            return super.retryRequest(exception, executionCount, context);
        }
    }
}
