package ru.yandex.direct.chassis.util

import org.slf4j.Logger
import org.slf4j.LoggerFactory
import ru.yandex.direct.env.Environment
import ru.yandex.direct.tracing.Trace
import java.lang.Thread.sleep
import java.time.Duration
import java.time.LocalDateTime
import java.time.format.DateTimeFormatter

object Utils {
    val logger: Logger = LoggerFactory.getLogger(Utils::class.java)

    fun retry(tries: Int, pause: Duration = Duration.ofSeconds(1), action: () -> Unit) {
        try {
            retryThrowing(tries, pause, action)
        } catch (e: InterruptedException) {
            Thread.currentThread().interrupt()
            throw e
        } catch (ignored: Exception) {
        }
    }

    fun <T> retryThrowing(tries: Int, pause: Duration = Duration.ofSeconds(1), action: () -> T): T {
        check(tries > 0) { "Try count has to be positive" }

        for (i in 0..tries) {
            try {
                return action()
            } catch (e: InterruptedException) {
                Thread.currentThread().interrupt()
                throw e
            } catch (e: Exception) {
                if (i < tries - 1) {
                    logger.warn("Exception on action, will be retried.", e)
                    sleep(pause.toMillis())
                } else {
                    logger.error("Exception on action, all retries exhausted.", e)
                    throw e
                }
            }
        }

        throw AssertionError("Unreachable")
    }

    private val logViewerDateTimeFormatter: DateTimeFormatter = DateTimeFormatter.ofPattern("yyyyMMdd'T'HHmmss")

    fun logViewerLink(from: LocalDateTime, to: LocalDateTime, service: String, traceId: Long): String {
        val domain = if (Environment.getCached().isProductionOrPrestable)
            "direct.yandex.ru" else "test.direct.yandex.ru"

        val fromString = logViewerDateTimeFormatter.format(from)
        val toString = logViewerDateTimeFormatter.format(to)

        return "https://$domain/logviewer#~(form~(from~'$fromString~to~'$toString" +
                "~fields~(~'log_time~'host~'service~'method~'trace_id~'span_id~'prefix~'log_level~'class_name" +
                "~'message)~conditions~(service~'$service~trace_id~'$traceId)~limit~100~offset~0" +
                "~reverseOrder~false~showTraceIdRelated~false)~logType~'messages)\$"
    }

    fun currentLogViewerLink() : String {
        val trace = Trace.current()
        val from = LocalDateTime.now().minusDays(1)
        val to = LocalDateTime.now().plusDays(1)

        return logViewerLink(from, to, trace.service, trace.traceId)
    }

    private val DIRECT_ISSUE_REGEX = "DIRECT-\\d+".toRegex()

    fun getTicketByCommitMessage(message: String?): String? {
        if (message == null) {
            return null
        }
        return DIRECT_ISSUE_REGEX.find(message)?.groupValues?.get(0)
    }
}
