package ru.yandex.direct.chassis.util;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.time.Duration;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;

import com.google.protobuf.ByteString;
import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.jetbrains.annotations.Nullable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.inside.yt.kosher.impl.ytree.serialization.YTreeTextSerializer;
import ru.yandex.yp.YpInstance;
import ru.yandex.yp.YpRawClient;
import ru.yandex.yp.YpRawClientBuilder;
import ru.yandex.yp.client.api.TMultiClusterReplicaSetSpec;
import ru.yandex.yp.client.api.TStageSpec;
import ru.yandex.yp.model.YpGetStatement;
import ru.yandex.yp.model.YpObjectType;
import ru.yandex.yp.model.YpPayload;
import ru.yandex.yp.model.YpSelectStatement;
import ru.yandex.yp.model.YpTypedId;

public class YpClient {

//    public static void main(String[] args) {
//        YpClient client = new YpClient(ru.yandex.direct.liveresource.LiveResourceFactory.get("file://~/.yp/token")
//                .getContent()
//                .trim());
//        List<String> allHosts = client.getHosts("direct-java-jobs-test");
//        System.out.println(allHosts);
//    }

    private static final Logger logger = LoggerFactory.getLogger(YpClient.class);
    public static final Duration YP_API_TIMEOUT = Duration.ofSeconds(10);

    private final Map<YpInstance, YpRawClient> clients;

    public YpClient(String token) {
        clients = new HashMap<>();
        clients.put(YpInstance.CROSS_DC, new YpRawClientBuilder(YpInstance.CROSS_DC, () -> token).build());
        clients.put(YpInstance.IVA, new YpRawClientBuilder(YpInstance.IVA, () -> token).build());
        clients.put(YpInstance.MAN, new YpRawClientBuilder(YpInstance.MAN, () -> token).build());
        clients.put(YpInstance.SAS, new YpRawClientBuilder(YpInstance.SAS, () -> token).build());
        clients.put(YpInstance.VLA, new YpRawClientBuilder(YpInstance.VLA, () -> token).build());
    }

    public List<String> getHosts(String stage) {
        var clusters = getClusters(stage);
        var hosts = EntryStream.of(clusters)
                .flatMapValues(Collection::stream)
                .mapToValue((unit, yp) -> getHosts(yp, stage, unit))
                .flatMapValues(Collection::stream)
                .grouping();

        // тут в будущем может быть прикручена фильтрация по deploy-unit
        return StreamEx.ofValues(hosts)
                .flatMap(Collection::stream)
                .toList();
    }

    private Map<String, List<YpInstance>> getClusters(String stage) {
        var statement = YpGetStatement.protobufBuilder(new YpTypedId(stage, YpObjectType.STAGE))
                .addSelector("/spec")
                .build();
        try {
            return clients.get(YpInstance.CROSS_DC)
                    .objectService()
                    .getObject(statement, this::parseClusters)
                    .get(YP_API_TIMEOUT.getSeconds(), TimeUnit.SECONDS);
        } catch (InterruptedException | ExecutionException | TimeoutException e) {
            if (e instanceof InterruptedException) {
                Thread.currentThread().interrupt();
            }
            logger.warn("Failed to get clusters for stage " + stage, e);
            return Collections.emptyMap();
        }
    }

    private Map<String, List<YpInstance>> parseClusters(List<YpPayload> ypPayloads) {
        try {
            Optional<ByteString> value = ypPayloads.get(0).getProtobuf();
            if (value.isEmpty()) {
                return Collections.emptyMap();
            }
            var deployUnits = TStageSpec.parseFrom(value.get()).getDeployUnitsMap();
            return EntryStream.of(deployUnits)
                    .mapValues(unit -> unit.getMultiClusterReplicaSet().getReplicaSet().getClustersList())
                    .mapValues(list -> StreamEx.of(list)
                            .map(TMultiClusterReplicaSetSpec.TClusterReplicaSetSpecPreferences::getCluster)
                            .map(this::toYpInstance)
                            .nonNull()
                            .toList())
                    .toMap();
        } catch (IOException e) {
            logger.warn("Failed to parse clusters", e);
            return Collections.emptyMap();
        }
    }

    private List<String> getHosts(YpInstance ypInstance, String stage, String unit) {
        var statement = YpSelectStatement.ysonBuilder(YpObjectType.POD)
                .addSelector("/status/dns/persistent_fqdn")
                .setFilter("[/meta/pod_set_id]='" + stage + "." + unit + "'")
                .build();
        try {
            return clients.get(ypInstance)
                    .objectService()
                    .selectObjects(statement, p -> parseString(p.get(0)))
                    .get(YP_API_TIMEOUT.getSeconds(), TimeUnit.SECONDS)
                    .getResults();
        } catch (InterruptedException | ExecutionException | TimeoutException e) {
            if (e instanceof InterruptedException) {
                Thread.currentThread().interrupt();
            }
            logger.warn("Failed to get hosts for pod " + stage + "." + unit, e);
            return Collections.emptyList();
        }
    }

    private static String parseString(YpPayload payload) {
        var input = new ByteArrayInputStream(payload.getYson().orElseThrow().toByteArray());
        var node = YTreeTextSerializer.deserialize(input);
        return node.isStringNode() ? node.stringValue() : null;
    }

    @Nullable
    private YpInstance toYpInstance(String cluster) {
        switch (cluster) {
            case "iva":
                return YpInstance.IVA;
            case "man":
                return YpInstance.MAN;
            case "sas":
                return YpInstance.SAS;
            case "vla":
                return YpInstance.VLA;
            default:
                logger.warn("Unknown cluster " + cluster);
                return null;
        }
    }
}
