package ru.yandex.direct.chassis.util.abc

import io.netty.handler.codec.http.HttpHeaders
import org.asynchttpclient.AsyncHttpClient
import org.springframework.http.MediaType
import org.springframework.stereotype.Component
import ru.yandex.direct.asynchttp.FetcherSettings
import ru.yandex.direct.asynchttp.ParallelFetcherFactory
import ru.yandex.direct.http.smart.annotations.Json
import ru.yandex.direct.http.smart.core.Call
import ru.yandex.direct.http.smart.core.Smart
import ru.yandex.direct.http.smart.error.ErrorUtils
import ru.yandex.direct.http.smart.http.GET
import ru.yandex.direct.http.smart.http.Headers
import ru.yandex.direct.http.smart.http.Query
import ru.yandex.direct.tvm.TvmIntegration
import ru.yandex.direct.tvm.TvmService

private const val SERVICE_SLUG = "direct"

/**
 * Клиент для ABC
 *
 * [Актуальная документация](https://wiki.yandex-team.ru/intranet/abc/docs/api/)
 */
@Component
class AbcClient(
    asyncHttpClient: AsyncHttpClient,
    tvmIntegration: TvmIntegration
) {
    private val baseUrl = "https://abc-back.yandex-team.ru/api/v4/"

    private val api: AbcApi = Smart.builder()
        .useTvm(tvmIntegration, TvmService.ABC_PROD)
        .withParallelFetcherFactory(ParallelFetcherFactory(asyncHttpClient, FetcherSettings()))
        .withProfileName("abc_client")
        .withBaseUrl(baseUrl)
        .addHeaderConfigurator { headers: HttpHeaders ->
            headers.add(org.springframework.http.HttpHeaders.CONTENT_TYPE,
                MediaType.APPLICATION_JSON_VALUE)
        }
        .build()
        .create(AbcApi::class.java)

    /**
     * Возвращает логины текущих дежурных в соответствующем расписании
     */
    fun getCurrentDutyLogins(serviceSlug: ServiceSlug, scheduleSlug: ScheduleSlug): List<Person> {
        return api.getShifts(serviceSlug.slug, scheduleSlug.slug)
            .executeCall()
            .map { it.person }
    }

    private fun <T> Call<T>.executeCall(): T {
        val result = execute()
        ErrorUtils.checkResultForErrors(result, { RuntimeException(it) })
        return result.success
    }
}

private interface AbcApi {
    @GET("/duty/on_duty/")
    @Json
    @Headers("Content-Type: application/json")
    fun getShifts(
        @Query("service__slug") serviceSlug: String,
        @Query("schedule__slug") scheduleSlug: String,
        @Query("with_watcher") withWatcher: Int = 1,
    ): Call<List<Shift>>
}
