package ru.yandex.direct.chassis.util.aqua

import com.fasterxml.jackson.databind.DeserializationFeature
import com.fasterxml.jackson.databind.ObjectMapper
import org.asynchttpclient.AsyncHttpClient
import org.springframework.stereotype.Component
import ru.yandex.direct.asynchttp.FetcherSettings
import ru.yandex.direct.asynchttp.ParallelFetcherFactory
import ru.yandex.direct.http.smart.annotations.Json
import ru.yandex.direct.http.smart.annotations.ResponseHandler
import ru.yandex.direct.http.smart.converter.CustomJsonResponseConverter
import ru.yandex.direct.http.smart.converter.ResponseConverterFactory
import ru.yandex.direct.http.smart.core.Call
import ru.yandex.direct.http.smart.core.Smart
import ru.yandex.direct.http.smart.error.ErrorUtils.checkResultForErrors
import ru.yandex.direct.http.smart.http.DELETE
import ru.yandex.direct.http.smart.http.GET
import ru.yandex.direct.http.smart.http.Headers
import ru.yandex.direct.http.smart.http.PUT
import ru.yandex.direct.http.smart.http.Path
import ru.yandex.direct.http.smart.http.Query
import ru.yandex.direct.utils.JsonUtils

object AquaUtils {
    const val AQUA_URL = "https://aqua.yandex-team.ru"
    const val AQUA_API_URL = "$AQUA_URL/aqua-api/services"

    fun getLaunchesUrl(tag: String) = "$AQUA_URL/#/launches-tag?tag=$tag"
}

@Component
class AquaClient(
    asyncHttpClient: AsyncHttpClient,
) {

    private val api: AquaApi = Smart.builder()
        .withResponseConverterFactory(
            ResponseConverterFactory.builder()
                .addConverters(AquaResponseConverter())
                .build()
        )
        .withParallelFetcherFactory(ParallelFetcherFactory(asyncHttpClient, FetcherSettings()))
        .withProfileName("aqua_client")
        .withBaseUrl(AquaUtils.AQUA_API_URL)
        .build()
        .create(AquaApi::class.java)

    fun getLaunches(tag: String): GetLaunchesResponse =
        api.getLaunches(tag).executeCall()

    fun restartLaunch(launchId: String, failedOnly: Boolean = true): RestartResponse =
        api.restartLaunch(launchId, failedOnly).executeCall()

    fun addLaunchTag(launchId: String, tag: String): Void =
        api.addLaunchTag(launchId, tag).executeCall()

    fun getLaunch(launchId: String): Launch =
        api.getLaunch(launchId).executeCall()

    fun getPackLaunches(
        skip: Int, limit: Int, packId: String,
        user: String = "", failedOnly: Boolean = false
    ): GetLaunchesResponse =
        api.getPackLaunches(skip, limit, packId, user, failedOnly)
            .executeCall()

    fun deleteLaunch(launchId: String): Void =
        api.deleteLaunch(launchId).executeCall()

    private fun <T> Call<T>.executeCall(): T {
        val result = execute()
        checkResultForErrors(result, { RuntimeException(it) })
        return result.success
    }
}

class AquaResponseConverter : CustomJsonResponseConverter<Any>() {
    override fun buildObjectMapper(): ObjectMapper = JsonUtils.MAPPER.copy()
        .disable(DeserializationFeature.READ_DATE_TIMESTAMPS_AS_NANOSECONDS)
}

interface AquaApi {
    @GET("/launch/tag/{tag}")
    @Headers("Content-Type: application/json")
    @ResponseHandler(parserClass = AquaResponseConverter::class)
    fun getLaunches(
        @Path("tag") tag: String,
    ): Call<GetLaunchesResponse>

    @GET("/launch/{launchId}/restart")
    @Json
    @Headers("Content-Type: application/json")
    fun restartLaunch(
        @Path("launchId") launchId: String,
        @Query("failed-only") failedOnly: Boolean,
    ): Call<RestartResponse>

    @PUT("/launch/addtag/{launchId}/{tag}")
    @Json
    @Headers("Content-Type: application/json")
    fun addLaunchTag(
        @Path("launchId") launchId: String,
        @Path("tag") tag: String,
    ): Call<Void>

    @GET("/launch/{launchId}")
    @Headers("Content-Type: application/json")
    @ResponseHandler(parserClass = AquaResponseConverter::class)
    fun getLaunch(
        @Path("launchId") launchId: String,
    ): Call<Launch>

    @DELETE("/launch/{launchId}")
    @Json
    @Headers("Content-Type: application/json")
    fun deleteLaunch(
        @Path("launchId") launchId: String,
    ): Call<Void>

    @GET("/launch/page/simple")
    @Headers("Content-Type: application/json")
    @ResponseHandler(parserClass = AquaResponseConverter::class)
    fun getPackLaunches(
        @Query("skip") skip: Int,
        @Query("limit") limit: Int,
        @Query("packId") packId: String,
        @Query("user") user: String,
        @Query("failed-only") failedOnly: Boolean,
    ): Call<GetLaunchesResponse>
}
