package ru.yandex.direct.chassis.util.github

import org.slf4j.LoggerFactory
import org.springframework.http.HttpHeaders
import org.springframework.http.HttpStatus
import org.springframework.http.ResponseEntity
import org.springframework.stereotype.Component
import org.springframework.util.LinkedMultiValueMap
import org.springframework.web.client.RestTemplate
import org.springframework.web.util.UriComponentsBuilder
import ru.yandex.direct.chassis.entity.startrek.AwaitingReleaseProcessor
import ru.yandex.direct.chassis.util.startrek.StartrekQueue

private const val SCHEME = "https"
private const val BASE_HOST = "github.yandex-team.ru"
private const val API_PREFIX = "api/v3"

/**
 * Клиент над GitHub REST API
 */
@Component
class GitHubClient(
    private val restTemplate: RestTemplate
) {

    companion object {
        private val logger = LoggerFactory.getLogger(AwaitingReleaseProcessor::class.java)
        private val PAGINATION_LINK_HEADER_REGEX = "page=(\\d+)>; rel=\"last\"".toRegex()
    }

    /**
     * Получить названия всех тикетов в формате DIRECT-******, по которым были коммиты от [from] до [to]
     *
     * [Получение коммитов](https://docs.github.com/en/free-pro-team@latest/rest/reference/repos#list-commits)
     * Используется [пагинация](https://docs.github.com/en/free-pro-team@latest/rest/guides/traversing-with-pagination)
     *
     * (!) [from] и [to] должны быть формата yyyy-MM-dd'T'HH:mm:ssZ -- это не валидируется в этом методе
     *
     */
    fun getCommittedDnaIssues(from: String, to: String): List<String> = try {
        val firstPageResponse = getCommits(from, to)
        if (firstPageResponse.statusCode != HttpStatus.OK) {
            throw RuntimeException("Status code = " + firstPageResponse.statusCode)
        }
        val dnaIssues = mutableListOf<String>()
        firstPageResponse.body?.getIssues()?.let { dnaIssues.addAll(it) }

        val totalPages = extractLastPage(firstPageResponse.headers)
        for (page in 2..totalPages) {
            getCommittedDnaIssues(from, to, page = page).let { dnaIssues.addAll(it) }
        }
        dnaIssues
    } catch (e: RuntimeException) {
        logger.error("Failed to get info from GitHub", e)
        emptyList()
    }

    private fun getCommittedDnaIssues(from: String, to: String, page: Int): List<String> = try {
        val response = getCommits(from, to, page)
        if (response.statusCode != HttpStatus.OK) {
            throw RuntimeException("Status code = " + response.statusCode)
        }
        response.body?.getIssues() ?: emptyList()
    } catch (e: RuntimeException) {
        logger.error("Failed to get info from GitHub", e)
        emptyList()
    }

    private fun getCommits(from: String, to: String, page: Int = 1): ResponseEntity<CommitResponse> {
        val params = LinkedMultiValueMap<String, String>()
        params.add("since", from)
        params.add("until", to)
        params.add("page", page.toString())
        val url = UriComponentsBuilder.newInstance()
            .scheme(SCHEME)
            .host(BASE_HOST)
            .path("$API_PREFIX/repos/direct/dna/commits")
            .queryParams(params)
            .build()
            .toUri()
            .toString()
        logger.info("Get commits info from GitHub: $url")
        return restTemplate.getForEntity(url, CommitResponse::class.java)
    }

    private fun extractLastPage(headers: HttpHeaders): Int {
        val links = headers["Link"] ?: return -1
        links.forEach {
            val lastPage = PAGINATION_LINK_HEADER_REGEX.find(it)?.groupValues?.get(1)?.toInt()
            if (lastPage != null) {
                return lastPage
            }
        }
        return -1
    }

}

class CommitResponse : ArrayList<Description>() {
    fun getIssues() = mapNotNull { it.commit?.extractIssue() }
}

data class Description(
    val commit: Commit? = null
)

data class Commit(
    val message: String? = null
) {
    companion object {
        private val DIRECT_ISSUE_REGEX = "^(${StartrekQueue.DIRECT}-\\d+)".toRegex()
    }

    fun extractIssue(): String? {
        if (message == null) {
            return null
        }
        message.split("\n").forEach {
            val get = DIRECT_ISSUE_REGEX.find(it)?.groupValues?.get(1)
            if (get != null) {
                return get
            }
        }
        return null
    }
}
