package ru.yandex.direct.chassis.util.jenkins

import org.apache.commons.codec.binary.Base64
import org.asynchttpclient.AsyncHttpClient
import org.springframework.http.HttpHeaders.AUTHORIZATION
import org.springframework.stereotype.Component
import ru.yandex.direct.asynchttp.FetcherSettings
import ru.yandex.direct.asynchttp.ParallelFetcherFactory
import ru.yandex.direct.http.smart.annotations.ResponseHandler
import ru.yandex.direct.http.smart.converter.VoidResponseConverter
import ru.yandex.direct.http.smart.core.Call
import ru.yandex.direct.http.smart.core.Smart
import ru.yandex.direct.http.smart.error.ErrorUtils.checkResultForErrors
import ru.yandex.direct.http.smart.http.POST
import ru.yandex.direct.http.smart.http.Path
import ru.yandex.direct.http.smart.http.QueryMap
import ru.yandex.direct.liveresource.LiveResource

@Component
class JenkinsClient(
    asyncHttpClient: AsyncHttpClient,
    private val user: String,
    private val password: LiveResource,
) {
    private val baseUrl = "https://jenkins-direct.qart.yandex-team.ru"

    private val authHeader
        get() = Base64.encodeBase64String("$user:${password.content}".toByteArray())

    private val api: JenkinsApi = Smart.builder()
        .withParallelFetcherFactory(ParallelFetcherFactory(asyncHttpClient, FetcherSettings()))
        .withProfileName("jenkins_client")
        .withBaseUrl(baseUrl)
        .addHeaderConfigurator { headers -> headers.add(AUTHORIZATION, "Basic $authHeader") }
        .build()
        .create(JenkinsApi::class.java)

    fun buildJob(job: String, params: Map<String, String>) =
        api.buildJob(job, params).executeCall()

    private fun <T> Call<T>.executeCall(): T {
        val result = execute()
        checkResultForErrors(result, { JenkinsException(it) })
        return result.success
    }
}

class JenkinsException(message: String) : RuntimeException(message)

interface JenkinsApi {
    @POST("/job/{job}/buildWithParameters")
    @ResponseHandler(expectedCodes = [201], parserClass = VoidResponseConverter::class)
    fun buildJob(
        @Path("job") job: String,
        @QueryMap params: Map<String, String>,
    ): Call<Void>
}
