package ru.yandex.direct.chassis.util.mysql

import com.fasterxml.jackson.databind.node.ObjectNode
import org.springframework.stereotype.Component
import ru.yandex.direct.config.DirectConfig
import ru.yandex.direct.db.config.DbConfigFactory
import ru.yandex.direct.dbutil.wrapper.DataSourceFactory
import ru.yandex.direct.dbutil.wrapper.DatabaseWrapperProvider
import ru.yandex.direct.dbutil.wrapper.ShardedDb
import ru.yandex.direct.env.EnvironmentType
import ru.yandex.direct.liveresource.LiveResource
import ru.yandex.direct.liveresource.LiveResourceEvent
import ru.yandex.direct.liveresource.LiveResourceListener
import ru.yandex.direct.liveresource.LiveResourceWatcherFactory
import ru.yandex.direct.liveresource.provider.LiveResourceFactoryBean
import ru.yandex.direct.utils.JsonUtils

data class MysqlUser(
    val user: String,
    val passwordYavPath: String
)

@Component
class TsShardsProviderFactory(
    private val cfg: DirectConfig,
    private val liveResourceFactory: LiveResourceFactoryBean,
    private val liveResourceWatcherFactory: LiveResourceWatcherFactory
) {

    fun readWrite() = TsShardsProvider(
        cfg, liveResourceFactory, liveResourceWatcherFactory,
        MysqlUser("direct-test", "yav://sec-01crg6rq6pgnw0fx47mc998p07?refreshRate=1800#mysql_direct-test")
    )

    fun readOnly() = TsShardsProvider(
        cfg, liveResourceFactory, liveResourceWatcherFactory,
        MysqlUser("direct-ro", "yav://sec-01crge0p4cryad9styvxqzqckm?refreshRate=1800#mysql_direct-ro")
    )
}

/**
 * Возвращает обертку DatabaseWrapperProvider для каждого шарда TS
 */
class TsShardsProvider(
    cfg: DirectConfig,
    liveResourceFactory: LiveResourceFactoryBean,
    liveResourceWatcherFactory: LiveResourceWatcherFactory,
    mysqlUser: MysqlUser,
) {

    private val dbConfigFactory: DbConfigFactory
    private val dbProvider: DatabaseWrapperProvider

    init {
        val zkDbConfig = liveResourceFactory[cfg.getString("db_config_ts_only")]
        val password = liveResourceFactory[mysqlUser.passwordYavPath]

        // творчески переделанный createAndWatch
        val resourceWatcher = liveResourceWatcherFactory.createWatcher(zkDbConfig)
        val patcher = DbConfigPatcher(password, mysqlUser.user)
        dbConfigFactory = DbConfigFactory(patcher.patchJson(resourceWatcher.watch()))
        resourceWatcher.addListener(ListenerProxy(dbConfigFactory, patcher))
        val dataSourceFactory = DataSourceFactory(cfg.getBranch("ts_datasource_factory_config"))
        dbProvider = DatabaseWrapperProvider.newInstance(dataSourceFactory, dbConfigFactory, EnvironmentType.TESTING)
    }

    fun get() = dbConfigFactory.getShardNumbers("ppc")
        .sorted()
        .map { dbProvider.get(ShardedDb.PPC, it) }
}

private class DbConfigPatcher(private val password: LiveResource, private val user: String) {
    fun patchJson(json: String?): String {
        val rootJson = JsonUtils.fromJson(json)
        val jsonNode = rootJson["db_config"]
        if (jsonNode.has("user")) {
            (jsonNode as ObjectNode).put("user", user)
        }
        if (jsonNode.has("pass")) {
            (jsonNode as ObjectNode).put("pass", password.content.trim { it <= ' ' })
        }
        if (jsonNode.has("extra_users")) {
            (jsonNode as ObjectNode).putObject("extra_users")
        }
        return JsonUtils.toJson(rootJson)
    }
}

private class ListenerProxy(private val originalListener: LiveResourceListener, private val patcher: DbConfigPatcher) :
    LiveResourceListener {
    override fun update(event: LiveResourceEvent) {
        val currentContent = event.currentContent
        val patchedContent = patcher.patchJson(currentContent)
        val patchedEvent = LiveResourceEvent(patchedContent)
        originalListener.update(patchedEvent)
    }
}
